<?php

namespace Drupal\paragraphs_clipboard\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\paragraphs\Entity\Paragraph;

/**
 * Service for handling access checks for clipboard operations.
 */
class ParagraphsClipboardAccess implements AccessInterface {

  /**
   * Access check for the "Paste from Clipboard" button.
   *
   * @param \Drupal\paragraphs\Entity\Paragraph $paragraph
   *   The paragraph entity.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition where the paragraph would be pasted.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(Paragraph $paragraph, FieldDefinitionInterface $field_definition) {
    // Check if the paragraph can be updated.
    $access = AccessResult::allowedIf($paragraph->access('update'));

    $type_access = $this->isAllowedParagraphType($paragraph, $field_definition) ?
      AccessResult::allowed()
      : AccessResult::forbidden();

    $access = $access->andIf($type_access);

    return $access;
  }

  /**
   * Checks if the clipboard's paragraph type is allowed for a specific field.
   *
   * @param \Drupal\paragraphs\Entity\Paragraph $paragraph
   *   The paragraph entity from the clipboard.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition where the paragraph would be pasted.
   *
   * @return bool
   *   TRUE if the paragraph type is allowed for the field, FALSE otherwise.
   */
  private function isAllowedParagraphType(Paragraph $paragraph, FieldDefinitionInterface $field_definition) {
    $handler_settings = $field_definition->getSetting('handler_settings');
    $target_bundles = $handler_settings['target_bundles'] ?? [];
    $negate = $handler_settings['negate'] ?? FALSE;

    $clipboard_paragraph_bundle = $paragraph->bundle();
    if ($negate || empty($target_bundles)) {
      return !in_array($clipboard_paragraph_bundle, $target_bundles, TRUE);
    }

    return in_array($clipboard_paragraph_bundle, $target_bundles, TRUE);
  }

}
