<?php

namespace Drupal\paragraphs_clipboard;

use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\paragraphs\Entity\Paragraph;

/**
 * Service for handling clipboard operations for Paragraphs entities.
 */
class ParagraphsClipboardService {

  const CLIPBOARD_KEY = 'copy_clipboard';
  const CLIPBOARD_UUID_KEY = 'copy_clipboard_uuid';

  /**
   * The temp store factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity repository service.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * Constructs a new ParagraphsClipboardService object.
   *
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The temp store factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   */
  public function __construct(PrivateTempStoreFactory $temp_store_factory, EntityTypeManagerInterface $entity_type_manager, EntityRepositoryInterface $entity_repository) {
    $this->tempStoreFactory = $temp_store_factory->get('paragraphs_clipboard');
    $this->entityTypeManager = $entity_type_manager;
    $this->entityRepository = $entity_repository;
  }

  /**
   * Stores paragraph ID and revision ID to clipboard.
   *
   * @param string $key
   *   The key for the clipboard data.
   * @param array $data
   *   The data to store in the clipboard.
   */
  public function setClipboardData($key, $data) {
    $this->tempStoreFactory->delete($key);
    $this->tempStoreFactory->set($key, $data);
  }

  /**
   * Retrieves clipboard data.
   *
   * @param string $key
   *   The key for the clipboard data.
   *
   * @return \Drupal\Core\TempStore\PrivateTempStoreFactory
   *   An array containing the 'paragraph_id' and 'revision_id'.
   */
  public function getClipboardData(string $key) {
    return $this->tempStoreFactory->get($key);
  }

  /**
   * Loads a paragraph revision.
   *
   * @param int $revision_id
   *   The ID of the paragraph revision to load.
   *
   * @return mixed
   *   The loaded paragraph entity object, or NULL if it cannot be loaded.
   */
  public function getParagraphByRevisionId($revision_id) {
    $paragraph_storage = $this->entityTypeManager->getStorage('paragraph');
    return $paragraph_storage->loadRevision($revision_id);
  }

  /**
   * Loads a paragraph by its UUID.
   *
   * @param string $uuid
   *   The UUID of the paragraph to load.
   *
   * @return \Drupal\paragraphs\Entity\Paragraph|null
   *   The loaded paragraph entity, or NULL if it cannot be loaded.
   */
  public function getParagraphByUuid(string $uuid) {
    return $this->entityRepository->loadEntityByUuid('paragraph', $uuid);
  }

  /**
   * Loads a paragraph by either revision ID or UUID.
   *
   * @param string $key
   *   The key to decide whether to load by UUID or revision ID.
   *
   * @return \Drupal\paragraphs\Entity\Paragraph|null
   *   The loaded paragraph entity, or NULL if it cannot be loaded.
   */
  public function getParagraphFromClipboard(string $key): ?Paragraph {
    $data = $this->getClipboardData($key);
    if (!$data) {
      return NULL;
    }

    if (isset($data['revision_id'])) {
      return $this->getParagraphByRevisionId($data['revision_id']);
    }
    if (isset($data['paragraph_uuid'])) {
      return $this->getParagraphByUuid($data['paragraph_uuid']);
    }

    return NULL;
  }

  /**
   * Check if the field has exceeded its cardinality limit.
   *
   * @param int $real_item_count
   *   The current count of items.
   * @param int $cardinality
   *   The cardinality of the field.
   *
   * @return bool
   *   TRUE if the field has space for more items, FALSE otherwise.
   */
  public function checkCardinality(int $real_item_count, int $cardinality) {
    return $real_item_count < $cardinality || $cardinality == FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED;
  }

}
