<?php

namespace Drupal\Tests\paragraphs_clipboard\Functional;

use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\Node;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\paragraphs_clipboard\ParagraphsClipboardService;
use Drupal\Tests\paragraphs\Functional\WidgetStable\ParagraphsTestBase;

/**
 * Tests paragraphs clipboard functionality.
 *
 * @group paragraphs_clipboard
 */
class ParagraphsClipboardTest extends ParagraphsTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'node',
    'paragraphs',
    'paragraphs_clipboard',
    'field_ui',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set default theme.
    \Drupal::service('theme_installer')->install(['claro']);
    $this->config('system.theme')->set('default', 'claro')->save();
  }

  /**
   * Tests the clipboard actions for paragraphs.
   */
  public function testCopyPasteButtons() {
    $this->addParagraphedContentType('paragraphed_test');

    // Add a Paragraph type with a text field.
    $this->addParagraphsType('text_paragraph');
    $this->addFieldtoParagraphType('text_paragraph', 'field_text', 'text');

    $this->loginAsAdmin([
      'create paragraphed_test content',
      'edit any paragraphed_test content',
    ]);

    // Create a text paragraph.
    $paragraph = Paragraph::create([
      'type' => 'text_paragraph',
      'field_text' => [
        'value' => 'Test text 1',
        'format' => 'plain_text',
      ],
    ]);
    $paragraph->save();

    // Add a node with the paragraphs.
    $node = Node::create([
      'type' => 'paragraphed_test',
      'title' => 'Test Node',
      'field_paragraphs' => [$paragraph],
    ]);
    $node->save();

    // Copy to clipboard.
    $this->drupalGet('node/' . $node->id() . '/edit');
    $this->submitForm([], 'field_paragraphs_0_copy_clipboard');

    $clipboard_service = \Drupal::service('paragraphs_clipboard.paragraphs_clipboard_service');
    $clipboard_data = $clipboard_service->getClipboardData(ParagraphsClipboardService::CLIPBOARD_KEY);

    // Assert that the clipboard data is correct.
    $this->assertEquals($paragraph->id(), $clipboard_data['paragraph_id'], 'The clipboard contains the correct paragraph ID.');
    $this->assertEquals($paragraph->getRevisionId(), $clipboard_data['revision_id'], 'The clipboard contains the correct revision ID.');

    // Paste from clipboard.
    $this->drupalGet('node/add/paragraphed_test');
    $this->submitForm([], 'field_paragraphs_paste_clipboard_add_more');
    $edit = [
      'title[0][value]' => 'clipboard_test',
    ];
    $this->submitForm($edit, 'Save');

    $node_clipboard = $this->getNodeByTitle('clipboard_test');
    $this->drupalGet('node/' . $node_clipboard->id());
    $this->assertSession()->pageTextContains('Test text 1');
  }

  /**
   * Tests the clipboard actions for paragraphs with a limited cardinality.
   */
  public function testCopyPasteButtonsWithLimitedCardinality() {
    $this->addParagraphedContentType('paragraphed_test');
    /** @var \Drupal\field\FieldStorageConfigInterface $field_storage */
    $field_storage = FieldStorageConfig::load('node.field_paragraphs');
    $field_storage->setCardinality(2)->save();

    $this->loginAsAdmin([
      'create paragraphed_test content',
      'edit any paragraphed_test content',
    ]);

    $this->addParagraphsType('text_paragraph');
    $this->addFieldtoParagraphType('text_paragraph', 'field_text', 'text');

    $this->loginAsAdmin([
      'create paragraphed_test content',
      'edit any paragraphed_test content',
    ]);

    // Create a text paragraph.
    $paragraph = Paragraph::create([
      'type' => 'text_paragraph',
      'field_text' => [
        'value' => 'Test text 1',
        'format' => 'plain_text',
      ],
    ]);
    $paragraph->save();

    // Add a node with the paragraphs.
    $node = Node::create([
      'type' => 'paragraphed_test',
      'title' => 'Test Node',
      'field_paragraphs' => [$paragraph],
    ]);
    $node->save();

    // Copy to clipboard.
    $this->drupalGet('node/' . $node->id() . '/edit');
    $this->submitForm([], 'field_paragraphs_0_copy_clipboard');

    $clipboard_service = \Drupal::service('paragraphs_clipboard.paragraphs_clipboard_service');
    $clipboard_data = $clipboard_service->getClipboardData(ParagraphsClipboardService::CLIPBOARD_KEY);

    // Assert that the clipboard data is correct.
    $this->assertEquals($paragraph->id(), $clipboard_data['paragraph_id'], 'The clipboard contains the correct paragraph ID.');
    $this->assertEquals($paragraph->getRevisionId(), $clipboard_data['revision_id'], 'The clipboard contains the correct revision ID.');

    // Paste from clipboard.
    $this->drupalGet('node/add/paragraphed_test');
    $this->submitForm([], 'field_paragraphs_paste_clipboard_add_more');
    $edit = [
      'title[0][value]' => 'clipboard_test_cardinality',
    ];
    $this->submitForm($edit, 'Save');

    $node_clipboard = $this->getNodeByTitle('clipboard_test_cardinality');
    $this->drupalGet('node/' . $node_clipboard->id() . '/edit');
    $this->assertSession()->buttonNotExists('field_paragraphs_paste_clipboard_add_more');
  }

}
