<?php

namespace Drupal\paragraphs_entity_embed;

use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for the custom embedded paragraph entities edit forms.
 */
class EmbeddedParagraphsForm extends ContentEntityForm {

  /**
   * The EmbeddedParagraphs entity.
   *
   * @var \Drupal\paragraphs_entity_embed\Entity\EmbeddedParagraphs
   */
  protected $entity;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $form = parent::create($container);
    $form->setMessenger($container->get('messenger'));
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {

    $form = parent::form($form, $form_state);
    $embed_paragraph = $this->entity;

    if ($this->operation == 'edit') {
      $form['#title'] = $this->t('Edit embed paragraph %label', ['%label' => $embed_paragraph->label()]);
    }
    $form['#attributes']['class'][0] = 'embed-paragraph-' . Html::getClass($embed_paragraph->bundle()) . '-form';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): void {
    $embed_paragraph = $this->entity;

    $insert = $embed_paragraph->isNew();
    $embed_paragraph->setNewRevision();
    $embed_paragraph->save();

    $logger = $this->logger('paragraphs_entity_embed');
    $t_args = [
      '@type' => $embed_paragraph->getEntityType()->getLabel(),
      '%info' => $embed_paragraph->uuid(),
    ];

    if ($insert) {
      $logger->notice('@type: added %info.', $t_args);
      $this->messenger->addMessage($this->t('@type %info has been created.', $t_args));
    }
    else {
      $logger->notice('@type: updated %info.', $t_args);
      $this->messenger->addMessage($this->t('@type %info has been updated.', $t_args));
    }
    if ($embed_paragraph->id()) {
      $form_state->setValue('id', $embed_paragraph->id());
      $form_state->set('id', $embed_paragraph->id());
    }
    else {
      // In the unlikely case something went wrong on save, the embed
      // paragraph will be rebuilt and embed paragraph redisplayed.
      $this->messenger->addMessage($this->t('Embeded Paragraph entity could not be saved.'), 'error');
      $form_state->setRebuild();
    }
  }

}
