<?php

namespace Drupal\paragraphs_entity_embed\Plugin\Field\FieldWidget;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\embed\Entity\EmbedButton;
use Drupal\paragraphs\Plugin\EntityReferenceSelection\ParagraphSelection;
use Drupal\paragraphs\Plugin\Field\FieldWidget\ParagraphsWidget;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'entity_reference embed paragraphs' widget.
 *
 * We hide add / remove buttons when translating to avoid accidental loss of
 * data because these actions effect all languages.
 */
#[FieldWidget(
  id: 'entity_reference_embed_paragraphs',
  label: new TranslatableMarkup('Embed Paragraphs'),
  description: new TranslatableMarkup('A embed paragraphs inline form widget.'),
  field_types: ['entity_reference_revisions'],
)]
class EmbedParagraphsWidget extends ParagraphsWidget {

  /**
   * Constructs a ParagraphsWidget object.
   *
   * @param string $plugin_id
   *   The plugin_id for the widget.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the widget is associated.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The bundle information.
   * @param \Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManagerInterface $selectionPluginManager
   *   The selection plugin manager.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    EntityFieldManagerInterface $entity_field_manager,
    protected EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    protected SelectionPluginManagerInterface $selectionPluginManager,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings, $entity_field_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('entity_field.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('plugin.manager.entity_reference_selection'),
    );
  }

  /**
   * Embed button that use this widget.
   *
   * @var \Drupal\embed\Entity\EmbedButton|array
   */
  protected EmbedButton|array $embedButton;

  /**
   * Returns the sorted allowed types for an entity reference field.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface|null $field_definition
   *   (optional) The field definition for which the allowed types should be
   *   returned, defaults to the current field.
   *
   * @return array
   *   A list of arrays keyed by the paragraph type machine name with the
   *   following properties.
   *     - label: The label of the paragraph type.
   *     - weight: The weight of the paragraph type.
   */
  public function getAllowedTypes(?FieldDefinitionInterface $field_definition = NULL): array {
    $return_bundles = [];
    /** @var \Drupal\Core\Entity\EntityReferenceSelection\SelectionPluginManagerInterface $selection_manager */
    $selection_manager = $this->selectionPluginManager;
    $handler = $selection_manager->getSelectionHandler($this->fieldDefinition);
    $bundles = $this->entityTypeBundleInfo->getBundleInfo($this->getFieldSetting('target_type'));

    if (!empty($this->embedButton->type_settings['enable_paragraph_type_filter'])) {
      $weight = 0;
      foreach ($this->embedButton->type_settings['paragraphs_type_filter'] as $bundle) {
        $return_bundles[$bundle] = [
          'label' => !empty($bundles[$bundle]) ? $bundles[$bundle]['label'] : $bundle,
          'weight' => $weight,
        ];
        $weight++;
      }
    }
    elseif ($handler instanceof ParagraphSelection) {
      $return_bundles = $handler->getSortedAllowedTypes();
    }
    // Support for other reference types.
    else {
      $weight = 0;
      foreach ($bundles as $machine_name => $bundle) {
        if (!count($this->getSelectionHandlerSetting('target_bundles')) || in_array($machine_name, $this->getSelectionHandlerSetting('target_bundles'))) {

          $return_bundles[$machine_name] = [
            'label' => $bundle['label'],
            'weight' => $weight,
          ];

          $weight++;
        }
      }
    }

    return $return_bundles;
  }

  /**
   * {@inheritdoc}
   */
  public function formMultipleElements(FieldItemListInterface $items, array &$form, FormStateInterface $form_state): array {
    // Set Embed button.
    $storage = $form_state->getStorage();
    $this->embedButton = $storage['editorParams']['embed_button'] ?? [];

    if (!empty($this->embedButton->type_settings['paragraphs_add_mode'])) {
      // Set the add mode specified.
      $this->settings['add_mode'] = $this->embedButton->type_settings['paragraphs_add_mode'];
    }

    return parent::formMultipleElements($items, $form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeaderActions(array $field_state, FormStateInterface $form_state): array {
    // Build header actions are not needed on embedded paragraphs.
    return [];
  }

}
