<?php

namespace Drupal\paragraphs_entity_embed\Controller;

use Drupal\Component\Utility\Tags;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Database;
use Drupal\Core\Entity\RevisionableStorageInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\editor\EditorInterface;
use Drupal\embed\EmbedButtonInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller that handle the CKEditor embed form for paragraphs.
 */
class ParagraphsEntityEmbedController extends ControllerBase {

  use LoggerChannelTrait;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $entity_manager = $container->get('entity_type.manager');
    return new static(
      $entity_manager->getStorage('embedded_paragraphs'),
      Database::getConnection()
    );
  }

  /**
   * Constructs a EmbeddedParagraphs object.
   *
   * @param \Drupal\Core\Entity\RevisionableStorageInterface $embeddedParagraphsStorage
   *   The custom embedded paragraphs storage.
   * @param \Drupal\Core\Database\Connection $connection
   *   Database connection.
   */
  public function __construct(protected RevisionableStorageInterface $embeddedParagraphsStorage, protected Connection $connection) {
  }

  /**
   * Presents the embedded paragraphs creation form.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request object.
   * @param \Drupal\editor\EditorInterface|null $editor
   *   The WYSIWYG editor.
   * @param \Drupal\embed\EmbedButtonInterface|null $embed_button
   *   The embed button.
   *
   * @return array
   *   A form array as expected by drupal_render().
   */
  public function addForm(Request $request, ?EditorInterface $editor = NULL, ?EmbedButtonInterface $embed_button = NULL): array {
    if (($return_html = $this->controllerCalledOutsideIframe($request))) {
      return $return_html;
    }

    $embedded_paragraphs = $this->embeddedParagraphsStorage->create([]);

    $form_state['editorParams'] = [
      'editor' => $editor,
      'embed_button' => $embed_button,
    ];

    return $this->entityFormBuilder()
      ->getForm($embedded_paragraphs, 'paragraphs_entity_embed', $form_state);
  }

  /**
   * Presents the embedded paragraphs update form.
   *
   * @param string $embedded_paragraphs_uuid
   *   The UUID of Embedded paragraphs we are going to edit via CKE modal form.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request object.
   * @param \Drupal\editor\EditorInterface|null $editor
   *   The WYSIWYG editor.
   * @param \Drupal\embed\EmbedButtonInterface|null $embed_button
   *   The embed button.
   * @param string $embedded_paragraphs_revision_id
   *   The revision id of the embedded paragraph to edit, defaults to latest.
   *
   * @return array
   *   A form array as expected by drupal_render().
   */
  public function editForm(
    string $embedded_paragraphs_uuid,
    Request $request,
    ?EditorInterface $editor = NULL,
    ?EmbedButtonInterface $embed_button = NULL,
    string $embedded_paragraphs_revision_id = 'latest',
  ): array {
    if (($return_html = $this->controllerCalledOutsideIframe($request))) {
      return $return_html;
    }

    if ($embedded_paragraphs_revision_id !== 'latest') {
      $embedded_paragraph = $this->embeddedParagraphsStorage
        ->loadRevision($embedded_paragraphs_revision_id);
    }
    else {
      $entity = $this->embeddedParagraphsStorage
        ->loadByProperties(['uuid' => $embedded_paragraphs_uuid]);

      $embedded_paragraph = current($entity);
    }

    $form_state['editorParams'] = [
      'editor' => $editor,
      'embed_button' => $embed_button,
      'align' => $request->query->get('align') ?? '',
    ];

    return $this->entityFormBuilder()
      ->getForm($embedded_paragraph, 'paragraphs_entity_embed', $form_state);
  }

  /**
   * Checks whether the current request is performed inside iframe.
   *
   * If its inside iframe nothing is returned, otherwise we return html markup
   * for showing iframe.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request object.
   *
   * @return array|null
   *   Array of the response data or nothing if the controller is called inside
   *   iframe.
   */
  private function controllerCalledOutsideIframe(Request $request): ?array {
    if (!$request->query->get('paragraphs_entity_embed_inside_iframe')) {
      $parsed_url = UrlHelper::parse($request->getRequestUri());
      if (isset($parsed_url['query']['_wrapper_format'])) {
        unset($parsed_url['query']['_wrapper_format']);
      }
      $parsed_url['query']['paragraphs_entity_embed_inside_iframe'] = 1;

      $iframe_source = $parsed_url['path'] . '?' . UrlHelper::buildQuery($parsed_url['query']);

      return [
        '#type' => 'html_tag',
        '#tag' => 'iframe',
        '#attributes' => [
          'src' => $iframe_source,
          'width' => '480',
          'height' => '100%',
          'frameBorder' => 0,
        ],
        '#attached' => ['library' => ['editor/drupal.editor.dialog']],
      ];
    }
    return NULL;
  }

  /**
   * Handler for autocomplete request.
   */
  public function handleAutocomplete(Request $request): JsonResponse {
    $results = [];
    // Get the typed string from the URL, if it exists.
    if ($input = $request->query->get('q')) {
      $typed_string = Tags::explode($input);
      $typed_string = mb_strtolower(array_pop($typed_string));
      $query = $this->connection->select('embedded_paragraphs', 'ep');
      $result = $query->fields('ep', ['uuid', 'label'])
        ->condition('ep.label', '%' . $query->escapeLike($typed_string) . '%', 'LIKE')
        ->orderBy('label')
        ->execute()
        ->fetchAll();

      foreach ($result as $item) {
        $results[] = [
          'value' => $item->uuid,
          'label' => $item->label,
        ];
      }
    }

    return new JsonResponse($results);
  }

  /**
   * Returns a page title.
   *
   * @param \Drupal\embed\EmbedButtonInterface|null $embed_button
   *   The embed button.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   Page title.
   */
  public function getEditTitle(?EmbedButtonInterface $embed_button = NULL): TranslatableMarkup {
    return $this->t('Edit %title', ['%title' => $embed_button->label()]);
  }

  /**
   * Returns a page title.
   *
   * @param \Drupal\embed\EmbedButtonInterface|null $embed_button
   *   The embed button.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   Page title.
   */
  public function getAddTitle(?EmbedButtonInterface $embed_button = NULL): TranslatableMarkup {
    return $this->t('Select %title to Embed', ['%title' => $embed_button->label()]);
  }

}
