<?php

namespace Drupal\paragraphs_entity_embed\Plugin\EmbedType;

use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\embed\EmbedType\EmbedTypeBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Paragraph embed type.
 *
 * @EmbedType(
 *   id = "paragraphs_entity_embed",
 *   label = @Translation("Paragraph")
 * )
 */
class Paragraph extends EmbedTypeBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a Paragraph object for paragraphs_entity_embed.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $fileUrlGenerator
   *   The file URL generator.
   * @param \Drupal\Core\Extension\ModuleExtensionList $moduleExtensionList
   *   The module extension list.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, protected FileUrlGeneratorInterface $fileUrlGenerator, protected ModuleExtensionList $moduleExtensionList, protected EntityTypeBundleInfoInterface $entityTypeBundleInfo) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('file_url_generator'),
      $container->get('extension.list.module'),
      $container->get('entity_type.bundle.info'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'enable_paragraph_type_filter' => FALSE,
      'paragraphs_type_filter' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultIconUrl(): string {
    $iconPath = $this->moduleExtensionList->getPath('paragraphs_entity_embed') . '/icons/paragraph.svg';
    return $this->fileUrlGenerator->generateAbsoluteString($iconPath);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {

    $form['enable_paragraph_type_filter'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Filter which Paragraph types to be embed'),
      '#default_value' => $this->getConfigurationValue('enable_paragraph_type_filter'),
    ];
    $form['paragraphs_type_filter'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Paragraph types'),
      '#default_value' => $this->getConfigurationValue('paragraphs_type_filter'),
      '#options' => $this->getAllParagraphTypes(),
      '#states' => [
        'visible' => [':input[name="type_settings[enable_paragraph_type_filter]"]' => ['checked' => TRUE]],
      ],
    ];
    $form['paragraphs_add_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Paragraphs add mode'),
      '#description' => $this->t('Select which add mode to use when selecting a paragraph, the default is a drop down.'),
      '#options' => [
        'dropdown' => $this->t('Dropdown select'),
        'button' => $this->t('Buttons'),
      ],
      '#default_value' => $this->getConfigurationValue('paragraphs_add_mode', 'dropdown'),
    ];

    return $form;
  }

  /**
   * Methods get all paragraph types as an option list.
   */
  protected function getAllParagraphTypes(): array {
    $paragraph_types = [];
    $types = $this->entityTypeBundleInfo->getBundleInfo('paragraph');
    foreach ($types as $machine_name => $type) {
      $paragraph_types[$machine_name] = $type['label'];
    }
    return $paragraph_types;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    if (!$form_state->hasAnyErrors()) {
      $this->setConfigurationValue('enable_paragraph_type_filter', $form_state->getValue('enable_paragraph_type_filter'));
      // Set views options.
      $paragraphs_types = $form_state->getValue('enable_paragraph_type_filter') ? array_filter($form_state->getValue('paragraphs_type_filter')) : [];
      $this->setConfigurationValue('paragraphs_type_filter', $paragraphs_types);

    }
  }

}
