<?php

namespace Drupal\paragraphs_entity_embed\Plugin\Filter;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Utility\Error;
use Drupal\embed\DomHelperTrait;
use Drupal\entity_embed\EntityEmbedBuilderInterface;
use Drupal\entity_embed\Exception\EntityNotFoundException;
use Drupal\entity_embed\Exception\RecursiveRenderingException;
use Drupal\filter\Attribute\Filter;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Drupal\filter\Plugin\FilterInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a filter to display embedded URLs based on data attributes.
 */
#[Filter(
  id: "paragraphs_entity_embed",
  title: new TranslatableMarkup("Display embedded paragraphs"),
  type: FilterInterface::TYPE_TRANSFORM_REVERSIBLE,
  description: new TranslatableMarkup("Embeds paragraphs using data attribute: data-paragraph-type."),
)]
class ParagraphEmbedFilter extends FilterBase implements ContainerFactoryPluginInterface {

  use DomHelperTrait;
  use LoggerChannelTrait;

  /**
   * Constructs a UrlEmbedFilter object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   Drupal renderer.
   * @param \Drupal\entity_embed\EntityEmbedBuilderInterface $builder
   *   Embed builder interface.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected RendererInterface $renderer,
    protected EntityEmbedBuilderInterface $builder,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('renderer'),
      $container->get('entity_embed.builder')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode): FilterProcessResult {
    $result = new FilterProcessResult($text);

    if (str_contains($text, 'data-paragraph-id')) {
      $dom = Html::load($text);
      $xpath = new \DOMXPath($dom);
      $entity_type = 'embedded_paragraphs';

      foreach ($xpath->query('//drupal-paragraph[@data-paragraph-id]') as $node) {
        $entity = NULL;
        $entity_output = '';

        try {
          // Load the entity either by UUID (preferred) or ID.
          $id = NULL;
          $id = $node->getAttribute('data-paragraph-id');
          $revision_id = $node->getAttribute('data-paragraph-revision-id');

          if (!empty($revision_id)) {
            /** @var \Drupal\Core\Entity\RevisionableStorageInterface $storage */
            $storage = $this->entityTypeManager->getStorage($entity_type);
            $entity = $storage->loadRevision($revision_id);
          }
          else {
            $embed_entity = $this->entityTypeManager->getStorage($entity_type)
              ->loadByProperties(['uuid' => $id]);
            $entity = current($embed_entity);
          }

          if ($entity) {
            // Protect ourselves from recursive rendering.
            static $depth = 0;
            $depth++;
            if ($depth > 20) {
              throw new RecursiveRenderingException(sprintf('Recursive rendering detected when rendering embedded %s entity %s.', $entity_type, $entity->id()));
            }
            $context = $this->getNodeAttributesAsArray($node);
            $context += ['data-langcode' => $langcode];
            $context['data-view-mode'] = 'embed';
            $context['data-entity-embed-display'] = 'entity_reference_revisions:entity_reference_revisions_entity_view';

            $build = $this->builder->buildEntityEmbed($entity, $context);

            // We need to render the embedded entity:
            // - without replacing placeholders, so that the placeholders are
            //   only replaced at the last possible moment. Hence, we cannot use
            //   either renderPlain() or renderRoot(), so we must use render().
            // - without bubbling beyond this filter, because filters must
            //   ensure that the bubbleable metadata for the changes they make
            //   when filtering text makes it onto the FilterProcessResult
            //   object that they return ($result). To prevent that bubbling, we
            //   must wrap the call to render() in a render context.
            $entity_output = $this->renderer->executeInRenderContext(new RenderContext(), function () use (&$build) {
              return $this->renderer->render($build);
            });
            $result = $result->merge(BubbleableMetadata::createFromRenderArray($build));

            $depth--;
          }
          else {
            throw new EntityNotFoundException(sprintf('Unable to load embedded %s entity %s.', $entity_type, $id));
          }
        }
        catch (RecursiveRenderingException | InvalidPluginDefinitionException | PluginNotFoundException | EntityNotFoundException $e) {
          if (method_exists(Error::class, 'logException')) {
            $this->getLogger('paragraphs_entity_embed')->error($e->getMessage());
          }
        }

        $this->replaceNodeContent($node, $entity_output);
      }

      $result->setProcessedText(Html::serialize($dom));
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function tips($long = FALSE) {
    if ($long) {
      return $this->t('
        <p>You can embed Drupal paragraph entities. Examples:</p>
        <ul>
          <li><code><drupal-paragraph data-paragraph-id="423d2d23d23-432423-432"> </drupal-paragraph> </code></li>
        </ul>');
    }
    else {
      return $this->t('You can embed Drupal Paragraphs.');
    }
  }

}
