<?php

namespace Drupal\paragraphs_field_validator\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\Entity\Node;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure paragraph field validator mappings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Constructor.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    parent::__construct($config_factory);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['paragraphs_field_validator.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'paragraphs_field_validator_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['description'] = [
      '#type' => 'item',
      '#markup' => $this->t('Configure paragraph bundles and fields to validate field formats.'),
    ];

    $config = $this->config('paragraphs_field_validator.settings');
    $mappings = $form_state->get('mappings') ?: $config->get('mappings') ?: [];
    $form_state->set('mappings', $mappings);

    $delta = $form_state->get('delta') ?? max(count($mappings), 1);
    $form_state->set('delta', $delta);

    if ($form_state->has('removed_delta')) {
      $mappings = array_values($form_state->get('mappings'));
      $form_state->set('mappings', $mappings);
    }

    $form['#tree'] = TRUE;
    $form['mappings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Validation rules'),
      '#prefix' => '<div id="mappings-wrapper">',
      '#suffix' => '</div>',
    ];

    // Build each mapping item.
    for ($i = 0; $i < $delta; $i++) {
      $form['mappings'][$i]['field_bundle'] = [
        '#type' => 'textfield',
        '#required' => TRUE,
        '#title' => $this->t('Paragraph bundle (machine name)'),
        '#default_value' => $mappings[$i]['field_bundle'] ?? NULL,
        '#description' => $this->t('Machine name of the paragraph bundle.'),
      ];
      $form['mappings'][$i]['field_key_machine_name'] = [
        '#type' => 'textfield',
        '#required' => TRUE,
        '#title' => $this->t('Key machine name'),
        '#default_value' => $mappings[$i]['field_key_machine_name'] ?? '',
        '#description' => $this->t('The machine name of the field used as "key".'),
      ];
      $form['mappings'][$i]['field_key_text'] = [
        '#type' => 'textfield',
        '#required' => TRUE,
        '#title' => $this->t('Key text'),
        '#default_value' => $mappings[$i]['field_key_text'] ?? '',
        '#description' => $this->t('The text used as key.'),
      ];
      $form['mappings'][$i]['field_value_machine_name'] = [
        '#type' => 'textfield',
        '#required' => TRUE,
        '#title' => $this->t('Value machine name'),
        '#default_value' => $mappings[$i]['field_value_machine_name'] ?? '',
        '#description' => $this->t('The machine name of the field used as "value".'),
      ];
      $form['mappings'][$i]['field_value_regex'] = [
        '#type' => 'textfield',
        '#required' => TRUE,
        '#title' => $this->t('Value regex'),
        '#default_value' => $mappings[$i]['field_value_regex'] ?? '',
        '#description' => $this->t('Regex validation for the value.'),
      ];
      $form['mappings'][$i]['field_regex_example'] = [
        '#type' => 'textfield',
        '#required' => TRUE,
        '#title' => $this->t('Regex example'),
        '#default_value' => $mappings[$i]['field_regex_example'] ?? '',
        '#description' => $this->t('An example value matching the regex.'),
      ];

      // Advanced config.
      $form['mappings'][$i]['advanced'] = [
        '#type' => 'details',
        '#title' => $this->t('Advanced config'),
        '#open' => FALSE,
      ];

      // Node selection options.
      $nodes = Node::loadMultiple();
      $options = [];
      foreach ($nodes as $node) {
        $options[$node->id()] = $node->label();
      }

      $form['mappings'][$i]['advanced']['apply_to_nodes'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Restrict to nodes'),
        '#description' => $this->t('Select nodes where this validation will apply. Leave empty to apply to all nodes.'),
        '#options' => $options,
        '#default_value' => $mappings[$i]['apply_to_nodes'] ?? [],
      ];

      if ($delta > 1) {
        $form['mappings'][$i]['remove_' . $i] = [
          '#type' => 'submit',
          '#value' => $this->t('Remove (@context)', ['@context' => $i + 1]),
          '#submit' => ['::removeCallback'],
          '#limit_validation_errors' => [],
          '#ajax' => [
            'callback' => '::ajaxCallbackSubmit',
            'wrapper' => 'mappings-wrapper',
          ],
          '#weight' => 100,
        ];
      }
    }

    $form['mappings']['actions'] = [
      '#type' => 'actions',
    ];
    $form['mappings']['actions']['add_more'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add one more'),
      '#submit' => ['::addOne'],
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => '::ajaxCallbackSubmit',
        'wrapper' => 'mappings-wrapper',
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    $mappings = $values['mappings'];
    unset($mappings['actions']);

    foreach ($mappings as $delta => $mapping) {
      $mappings[$delta] = [
        'field_bundle' => $mapping['field_bundle'],
        'field_key_machine_name' => $mapping['field_key_machine_name'],
        'field_key_text' => $mapping['field_key_text'],
        'field_value_machine_name' => $mapping['field_value_machine_name'],
        'field_value_regex' => $mapping['field_value_regex'],
        'field_regex_example' => $mapping['field_regex_example'],
        'apply_to_nodes' => array_filter($mapping['advanced']['apply_to_nodes'] ?? []),
      ];
    }

    $this->configFactory->getEditable('paragraphs_field_validator.settings')
      ->set('mappings', $mappings)
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Callback for ajax-enabled buttons.
   */
  public function ajaxCallbackSubmit(array &$form, FormStateInterface $form_state) {
    return $form['mappings'];
  }

  /**
   * Submit handler for the "Add one more" button.
   */
  public function addOne(array &$form, FormStateInterface $form_state) {
    $delta = (int) $form_state->get('delta');
    $form_state->set('delta', $delta + 1);
    $form_state->setRebuild();
  }

  /**
   * Submit handler for the "Remove" button.
   */
  public function removeCallback(array &$form, FormStateInterface $form_state) {
    $delta = (int) $form_state->get('delta');
    $mappings = $form_state->get('mappings') ?? [];

    if ($delta > 1) {
      $form_state->set('delta', $delta - 1);
      $triggering_element = $form_state->getTriggeringElement();
      [, $removed_delta] = $triggering_element['#parents'];
      unset($mappings[$removed_delta]);
      $mappings = array_values($mappings);
      $form_state->set('mappings', $mappings);

      $input = $form_state->getUserInput();
      $input['mappings'] = $mappings;
      $form_state->setUserInput($input);
    }

    $form_state->setRebuild();
  }

}
