<?php

namespace Drupal\paragraphs_field_validator\Plugin\Validation\Constraint;

use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;

/**
 * Applies configured validations to paragraph entities.
 */
class FieldFormatConstraintValidator extends ConstraintValidator {

  /**
   * {@inheritdoc}
   */
  public function validate($entity, Constraint $constraint) {
    if (!method_exists($entity, 'getEntityTypeId') || $entity->getEntityTypeId() !== 'paragraph') {
      return;
    }

    $config = \Drupal::config('paragraphs_field_validator.settings');
    $mappings = $config->get('mappings') ?: [];

    if (empty($mappings) || !is_array($mappings)) {
      return;
    }

    $nodeId = \Drupal::state()->get('paragraphs_field_validator.node_id');

    foreach ($mappings as $mapping) {
      if (!is_array($mapping)) {
        continue;
      }

      $fieldBundle      = $mapping['field_bundle'] ?? NULL;
      $fieldKeyId       = $mapping['field_key_machine_name'] ?? NULL;
      $fieldValueId     = $mapping['field_value_machine_name'] ?? NULL;
      $expectedKeyText  = strtolower(trim((string) ($mapping['field_key_text'] ?? '')));
      $pattern          = $mapping['field_value_regex'] ?? NULL;
      $regexExample     = $mapping['field_regex_example'] ?? '';
      $applyToNodes     = $mapping['apply_to_nodes'] ?? [];

      if (!$fieldBundle || !$fieldKeyId || !$fieldValueId || !$pattern) {
        continue;
      }

      if ($fieldBundle !== $entity->bundle()) {
        continue;
      }

      if (!empty($applyToNodes) && $nodeId && !in_array($nodeId, $applyToNodes, TRUE)) {
        continue;
      }

      if (!$entity->hasField($fieldKeyId) || !$entity->hasField($fieldValueId)) {
        continue;
      }

      $key   = strtolower((string) ($entity->get($fieldKeyId)->value ?? ''));
      $value = (string) ($entity->get($fieldValueId)->value ?? '');

      if ($key === $expectedKeyText && !preg_match($pattern, $value)) {
        $this->context->addViolation($constraint->message, [
          '@key' => $expectedKeyText,
          '@example' => $regexExample,
        ]);
        return;
      }
    }
  }
}
