<?php

namespace Drupal\Tests\paragraphs_field_validator\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field\Entity\FieldConfig;

/**
 * Kernel test for FieldFormatConstraintValidator.
 *
 * @group paragraphs_field_validator
 */
class FieldFormatConstraintValidatorTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'paragraphs_field_validator',
    'paragraphs',
    'node',
    'field',
    'user',
    'file',
    'text',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create paragraph type "text".
    ParagraphsType::create([
      'id' => 'text',
      'label' => 'Text',
    ])->save();

    // Create field storage for key and value fields.
    FieldStorageConfig::create([
      'field_name' => 'field_key',
      'entity_type' => 'paragraph',
      'type' => 'string',
    ])->save();

    FieldStorageConfig::create([
      'field_name' => 'field_value',
      'entity_type' => 'paragraph',
      'type' => 'string',
    ])->save();

    // Attach fields to the "text" paragraph type.
    FieldConfig::create([
      'field_name' => 'field_key',
      'entity_type' => 'paragraph',
      'bundle' => 'text',
      'label' => 'Key',
    ])->save();

    FieldConfig::create([
      'field_name' => 'field_value',
      'entity_type' => 'paragraph',
      'bundle' => 'text',
      'label' => 'Value',
    ])->save();

    // Configure the validator mappings.
    $this->container->get('config.factory')->getEditable('paragraphs_field_validator.settings')
      ->set('mappings', [
        [
          'field_bundle' => 'text',
          'field_key_machine_name' => 'field_key',
          'field_key_text' => 'foo',
          'field_value_machine_name' => 'field_value',
          'field_value_regex' => '/^[0-9]+$/',
          'field_regex_example' => '12345',
          'apply_to_nodes' => [],
        ],
      ])
      ->save();
  }

  /**
   * Tests that invalid values trigger a violation.
   */
  public function testValidationViolation(): void {
    // Paragraph with key "foo" and non-numeric value should fail.
    $paragraph = Paragraph::create([
      'type' => 'text',
      'field_key' => 'foo',
      'field_value' => 'abc',
    ]);
    $violations = $paragraph->validate();

    $this->assertGreaterThan(0, $violations->count(), 'Expected validation violation for non-numeric value.');
  }

  /**
   * Tests that valid values pass without violation.
   */
  public function testValidationPasses(): void {
    // Paragraph with key "foo" and numeric value should pass.
    $paragraph = Paragraph::create([
      'type' => 'text',
      'field_key' => 'foo',
      'field_value' => '12345',
    ]);
    $violations = $paragraph->validate();

    $this->assertCount(0, $violations, 'Expected no validation violations for numeric value.');
  }

}
