<?php

namespace Drupal\paragraphs_list_filter\Entity\ListBuilder;

use Drupal\Core\Database\Database;
use Drupal\Core\Url;
use Drupal\paragraphs\Controller\ParagraphsTypeListBuilder;
use Drupal\paragraphs\Entity\ParagraphsType;

/**
 *
 */
class ParagraphsListFilterBuilder extends ParagraphsTypeListBuilder {

  protected $currentPage;
  protected $totalCount;

  /**
   * {@inheritdoc}
   */
  public function render() {
    $this->limit = (Int) (\Drupal::request()->query->get('items_per_page') ?? 50);
    $this->currentPage = (Int) \Drupal::request()->query->get('page') ?? 0;
    $build['form'] = \Drupal::formBuilder()->getForm('Drupal\paragraphs_list_filter\Form\ParagraphsListFilterForm');
    $build += parent::render();
    $build['pager'] = $this->renderPager();
    $build['#attached']['library'][] = 'paragraphs_list_filter/paragraphs_list_filter.style';

    return $build;
  }

  /**
   *
   */
  public function load() {
    $label_filter = \Drupal::request()->query->get('label');
    $description_filter = \Drupal::request()->query->get('description');
    $id_filter = \Drupal::request()->query->get('id');
    $connection = Database::getConnection();
    $query = $connection->select('config', 'c')
      ->fields('c', ['name', 'data'])
      ->condition('name', '%paragraphs.paragraphs_type.%', 'LIKE');

    $results = $query->execute()->fetchAllKeyed();

    $this->setTotalCount(count($results));

    $filtered_entities = [];

    // Deserialize and filter in PHP.
    foreach ($results as $config_name => $serialized_data) {
      $data = unserialize($serialized_data);

      // Check if the label field contains the filter text.
      if ($id_filter != '' && preg_match("/^$id_filter$/i", $data['id'])) {
        $filtered_entities[] = $config_name;
      }
      elseif ($label_filter != '' && preg_match("/$label_filter/i", $data['label'])) {
        $filtered_entities[] = $config_name;
      }
      elseif ($description_filter != '' && preg_match("/$description_filter/i", $data['description'])) {
        $filtered_entities[] = $config_name;
      }
      elseif ($id_filter == '' && $label_filter == '' && $description_filter == '') {
        $filtered_entities[] = $config_name;
      }
    }

    $offset = $this->currentPage * $this->limit;
    $this->setTotalCount(count($filtered_entities));
    $filtered_entities = array_slice($filtered_entities, $offset, $this->limit);

    $entities = ParagraphsType::loadMultiple(array_map(function ($e) {
      return str_replace('paragraphs.paragraphs_type.', '', $e);

    }, $filtered_entities));

    return $entities;
  }

  /**
   * Custom method to render the pager with limited links.
   */
  public function renderPager() {
    $total_items = $this->getTotalCount();
    $total_pages = ceil($total_items / $this->limit);
    $output = [
      '#type' => 'ul',
      '#theme' => 'item_list',
      '#attributes' => [
        'id' => 'paragraph-type-list-filter',
        'class' => ['pager__items', 'js-pager__items'],
        'style' => 'list-style-type: none;',
      ],
    ];
    $output_li = [];

    // Build previous link if not on the first page.
    if ($this->currentPage > 0) {
      $output_li[] = [
        '#type' => 'link',
        '#title' => $this->t('« Previous'),
        '#url' => Url::fromRoute('entity.paragraphs_type.collection', ['page' => $this->currentPage - 1, 'items_per_page' => \Drupal::request()->get('items_per_page'), 'id' => \Drupal::request()->get('id'), 'label' => \Drupal::request()->get('label'), 'description' => \Drupal::request()->get('description')]),
        '#attributes' => [
          'class' => ['pager__item'],
        ],
      ];
    }

    // Calculate the range of pages to display.
    // Start from the current page - 1.
    $start = max(0, $this->currentPage - 1);
    // End at the current page + 1.
    $end = min($total_pages - 1, $this->currentPage + 1);

    // Adjust the start and end to ensure we display at most 3 links.
    if ($end - $start < 4) {
      if ($start > 0) {
        $start = max(0, $end - 2);
      }
      else {
        $end = min($total_pages - 1, $start + 2);
      }
    }

    // Build page links.
    for ($i = $start; $i <= $end; $i++) {
      $output_li[] = [
        '#type' => 'link',
      // Display page number (1-indexed)
        '#title' => (string) ($i + 1),
        '#url' => Url::fromRoute('entity.paragraphs_type.collection', ['page' => $i, 'items_per_page' => \Drupal::request()->get('items_per_page'), 'id' => \Drupal::request()->get('id'), 'label' => \Drupal::request()->get('label'), 'description' => \Drupal::request()->get('description')]),
        '#attributes' => ['class' => ['pager__item']],
      ];
    }

    // Build next link if not on the last page.
    if ($this->currentPage < $total_pages - 1) {
      $output_li[] = [
        '#type' => 'link',
        '#title' => $this->t('Next »'),
        '#url' => Url::fromRoute('entity.paragraphs_type.collection', ['page' => $this->currentPage + 1, 'items_per_page' => \Drupal::request()->get('items_per_page'), 'id' => \Drupal::request()->get('id'), 'label' => \Drupal::request()->get('label'), 'description' => \Drupal::request()->get('description')]),
        '#attributes' => ['class' => ['pager__item']],
      ];
    }
    $output['#items'] = $output_li;

    return $output;
  }

  /**
   * Set the total count of items.
   */
  protected function setTotalCount($count) {
    $this->totalCount = $count;
  }

  /**
   * Get the total count of items.
   */
  protected function getTotalCount() {
    return $this->totalCount ?? 0;
  }

}
