<?php

namespace Drupal\paragraphs_list_filter\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\paragraphs\Entity\ParagraphsType;

/**
 * Provides a filter form for the Paragraphs type list.
 *
 * This form allows administrators to filter the list of Paragraph types
 * by machine name, label, description and items per page.
 * It uses the GET method so filters are reflected in the URL and can
 * be bookmarked or shared.
 */
class ParagraphsListFilterForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'paragraphs_list_filter_form';
  }

  /**
   * {@inheritdoc}
   *
   * Builds the filter form elements.
   *
   * @param array $form
   *   The form render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   *
   * @return array
   *   The completed form render array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $request = $this->getRequest();

    // Use GET method to append parameters to the URL.
    $form['#method'] = 'get';
    $form['#attributes']['class'][] = 'paragraphs-list-filter-form';

    // Wrapper for filter fields.
    $form['filter'] = [
      '#type' => 'fieldset',
      '#attributes' => [
        'class' => ['form--inline', 'clearfix'],
      ],
    ];

    // Build select options for all available paragraph types.
    $options = ['' => ''];
    foreach (ParagraphsType::loadMultiple() as $pt) {
      $options[$pt->id()] = $pt->id();
    }

    // Filter by machine name (paragraph type ID).
    $form['filter']['id'] = [
      '#type' => 'select',
      '#title' => $this->t('Machine name'),
      '#options' => $options,
      '#default_value' => $request->query->get('id') ?? '',
    ];

    // Filter by label.
    $form['filter']['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#size' => 30,
      '#default_value' => $request->query->get('label') ?? '',
    ];

    // Filter by description.
    $form['filter']['description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Description'),
      '#size' => 30,
      '#default_value' => $request->query->get('description') ?? '',
    ];

    // Items per page dropdown.
    $form['filter']['limit'] = [
      '#type' => 'select',
      '#title' => $this->t('Per page'),
      '#options' => [
        10 => 10,
        20 => 20,
        30 => 30,
        50 => 50,
        100 => 100,
        200 => 200,
        500 => 500,
      ],
      '#default_value' => $request->query->get('limit') ?? 50,
    ];

    // Submit button to apply filters.
    $form['filter']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Filter'),
      '#attributes' => ['class' => ['button', 'button--primary']],
    ];

    // Reset link clears all filters and redirects to the base listing.
    $form['filter']['reset'] = [
      '#type' => 'link',
      '#title' => $this->t('Reset'),
      '#url' => Url::fromRoute('entity.paragraphs_type.collection'),
      '#attributes' => ['class' => ['button', 'button--secondary']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * Handles submission of the filter form.
   *
   * Redirects to the same listing route with the selected filter parameters
   * applied as query string arguments.
   *
   * @param array $form
   *   The form render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Collect input and build query string for redirect.
    $input = $form_state->getUserInput();
    $query = array_intersect_key($input, array_flip([
      'id',
      'label',
      'description',
      'limit',
    ]));

    $form_state->setRedirect('entity.paragraphs_type.collection', [], ['query' => $query]);
  }

}
