<?php

namespace Drupal\paragraphs_source_edit\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityFormBuilderInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\paragraphs\Entity\Paragraph;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

final class SourceEditController extends ControllerBase {

  /** @var \Drupal\Core\Entity\EntityFormBuilderInterface */
  protected EntityFormBuilderInterface $formBuilderSvc;

  public function __construct(EntityFormBuilderInterface $form_builder) {
    $this->formBuilderSvc = $form_builder;
  }

  public static function create(ContainerInterface $container): self {
    return new self($container->get('entity.form_builder'));
  }

  /**
   * Access: allow edit for both source and non-source languages,
   * still honoring normal update access.
   */
  public static function access(Paragraph $paragraph, LanguageInterface $language, AccountInterface $account): AccessResult {
    $update = $paragraph->access('update', $account, TRUE);

    // Validate the language object (should always be valid if param converter ran).
    $lang_ok = $language ? AccessResult::allowed() : AccessResult::forbidden();

    return $update
      ->andIf($lang_ok)
      ->addCacheableDependency($paragraph)
      ->cachePerPermissions();
  }

  /**
   * Controller: render the edit form in the requested language context.
   * If requested language is the source, render the base edit form.
   */
  public function edit(Paragraph $paragraph, LanguageInterface $language, Request $request) {
    $langcode = $language->getId();
    $source_lang = $paragraph->getUntranslated()->language()->getId();

    // Get entity in requested translation context.
    if ($langcode === $source_lang) {
      $entity_in_lang = $paragraph->getUntranslated();
    }
    else {
      $entity_in_lang = $paragraph->hasTranslation($langcode)
        ? $paragraph->getTranslation($langcode)
        : $paragraph->addTranslation($langcode);
    }

    // Use the normal entity edit form, bound to that translation.
    return $this->formBuilderSvc->getForm($entity_in_lang, 'edit');
  }

}

