<?php

namespace Drupal\paragraphs_source_edit;

use Drupal\content_translation\ContentTranslationHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Safer translation handler for Paragraphs.
 *
 * Avoids calling setAuthor()/setChanged() on fields that do not exist.
 * We also do NOT try to set the current langcode here; the controller
 * already ensures the entity is in the correct translation context.
 */
class ParagraphTranslationHandler extends ContentTranslationHandler {

  /**
   * {@inheritdoc}
   */
  public function entityFormEntityBuild($entity_type, EntityInterface $entity, array $form, FormStateInterface $form_state) {
    // Minimal metadata: set source only (this method exists on the wrapper).
    $wrapper = $this->manager->getTranslationMetadata($entity);
    $wrapper->setSource($entity->getUntranslated()->language()->getId());

    // Guard: only set author if the entity actually has an author field.
    $author_key = $entity->getEntityType()->getKey('uid');
    if ($author_key && $entity->hasField($author_key)) {
      $wrapper->setAuthor(\Drupal::currentUser()->id());
    }

    // Guard: only set changed time if the entity has a changed field.
    $changed_key = $entity->getEntityType()->getKey('changed');
    if ($changed_key && $entity->hasField($changed_key)) {
      $wrapper->setChangedTime(time());
    }

    // Do NOT call parent::entityFormEntityBuild() — it assumes fields like 'uid'
    // exist and can reintroduce the fatal on Paragraphs.
    return $form;
  }

}

