<?php

namespace Drupal\paragraphs_source_edit\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityFormBuilderInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\paragraphs\Entity\Paragraph;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for editing Paragraph entities in source and translated languages.
 *
 * Provides access checks and renders the edit forms for Paragraphs
 * in the requested language context.
 */
final class SourceEditController extends ControllerBase {

  /**
   * The entity form builder service.
   *
   * @var \Drupal\Core\Entity\EntityFormBuilderInterface
   */
  protected EntityFormBuilderInterface $formBuilderSvc;

  /**
   * The language manager service.
   *
   * Provides access to language information and operations.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Constructs a new SourceEditController object.
   *
   * @param \Drupal\Core\Entity\EntityFormBuilderInterface $form_builder
   *   The entity form builder service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager service.
   */
  public function __construct(EntityFormBuilderInterface $form_builder, LanguageManagerInterface $language_manager) {
    $this->formBuilderSvc = $form_builder;
    $this->languageManager = $language_manager;
  }

  /**
   * Access: allow edit for both source and non-source languages.
   *
   * Still honoring normal update access.
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('entity.form_builder'),
      $container->get('language_manager')
    );
  }

  /**
   * Access for the standard edit-form route.
   */
  public static function accessCurrent(Paragraph $paragraph, AccountInterface $account): AccessResult {
    return $paragraph->access('update', $account, TRUE)
      ->addCacheableDependency($paragraph)
      ->cachePerPermissions();
  }

  /**
   * Allow edit for both source and non-source languages.
   */
  public static function access(Paragraph $paragraph, LanguageInterface $language, AccountInterface $account): AccessResult {
    $update = $paragraph->access('update', $account, TRUE);

    // Validate the language object (should be valid if param converter ran).
    $lang_ok = $language ? AccessResult::allowed() : AccessResult::forbidden();

    return $update
      ->andIf($lang_ok)
      ->addCacheableDependency($paragraph)
      ->cachePerPermissions();
  }

  /**
   * Controller: render the edit form in the requested language context.
   *
   * If requested language is the source, render the base edit form.
   */
  public function edit(Paragraph $paragraph, LanguageInterface $language, Request $request) {
    $langcode = $language->getId();
    $source_lang = $paragraph->getUntranslated()->language()->getId();

    // Get entity in requested translation context.
    if ($langcode === $source_lang) {
      $entity_in_lang = $paragraph->getUntranslated();
    }
    else {
      $entity_in_lang = $paragraph->hasTranslation($langcode)
        ? $paragraph->getTranslation($langcode)
        : $paragraph->addTranslation($langcode);
    }

    // Use the normal entity edit form, bound to that translation.
    return $this->formBuilderSvc->getForm($entity_in_lang, 'edit');
  }

  /**
   * Edit in the current content language (backs toUrl('edit-form')).
   */
  public function editCurrent(Paragraph $paragraph, Request $request) {
    // Determine the desired language from the content language context.
    $langcode = $this->languageManager
      ->getCurrentLanguage(LanguageInterface::TYPE_CONTENT)
      ->getId();

    $source = $paragraph->getUntranslated()->language()->getId();

    // Pick the correct translation in context.
    if ($langcode === $source) {
      $entity_in_lang = $paragraph->getUntranslated();
    }
    else {
      $entity_in_lang = $paragraph->hasTranslation($langcode)
        ? $paragraph->getTranslation($langcode)
        : $paragraph->addTranslation($langcode);
    }

    return $this->formBuilderSvc->getForm($entity_in_lang, 'edit');
  }

}
