<?php

namespace Drupal\paragraphs_title_manager\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\Core\Database\Database;
use Drupal\paragraphs_title_manager\Helper\ParagraphsTitleManagerHelper;

/**
 * Provides a configurable settings form for Paragraphs Title Manager.
 *
 * This form:
 *  - Detects all paragraph bundles.
 *  - Auto-detects all title-like fields.
 *  - Allows enabling and aligning title fields.
 *  - Loads alignment options dynamically from DB (helper).
 *  - Shows UI notices when fields do not exist.
 *  - Provides responsive and modern UI layout.
 *  - Includes safe DB interactions using try/catch.
 */
class ParagraphsTitleManagerForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'paragraphs_title_manager_settings_form';
  }

  /**
   * Build the configuration form.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    // ------------------------------------------------------------
    // SAFE DB LOAD: Load saved settings.
    // ------------------------------------------------------------
    $saved = [];
    $has_saved_rows = FALSE;

    try {
      $connection = Database::getConnection();

      $saved_raw = $connection->select('paragraphs_title_manager', 'ptm')
        ->fields('ptm')
        ->execute()
        ->fetchAll();

      if (!empty($saved_raw)) {
        $has_saved_rows = TRUE;
      }

      foreach ($saved_raw as $row) {
        $saved[$row->pid][$row->field_name] = $row;
      }
    }
    catch (\Exception $e) {
      \Drupal::messenger()->addError('Error loading title alignment settings.');
    }

    // ------------------------------------------------------------
    // GLOBAL WRAPPER.
    // ------------------------------------------------------------
    $form['all_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'ptm-global-wrapper'],
      '#tree' => TRUE,
    ];

    // If DB/settings are completely empty, show a top intro hint.
    if (!$has_saved_rows) {
      $form['all_wrapper']['intro_message'] = [
        '#markup' => '<div class="ptm-info-message ptm-info-global">
          No existing title alignment settings were found. 
          Any Paragraph fields whose machine name contains <code>title</code> 
          (for example: <code>title</code>, <code>title_one</code>, <code>sub_title</code>) 
          will automatically appear below once they exist in your paragraph types.
        </div>',
      ];
    }

    // ------------------------------------------------------------
    // SELECT ALL / DESELECT ALL BUTTON.
    // ------------------------------------------------------------
    $form['all_wrapper']['ptm_global_toggle'] = [
      '#type' => 'button',
      '#value' => 'Select All',
      '#attributes' => [
        'class' => ['ptm-select-all-button', 'button'],
        'data-label-select' => 'Select All',
        'data-label-deselect' => 'Deselect All',
      ],
      '#prefix' => '<div class="ptm-global-toggle-btn">',
      '#suffix' => '</div>',
    ];

    // ------------------------------------------------------------
    // GLOBAL ALIGNMENT SELECTOR.
    // ------------------------------------------------------------
    $form['all_wrapper']['ptm_global_alignment'] = [
      '#type' => 'select',
      '#title' => 'Alignment for selected fields',
      '#options' => ParagraphsTitleManagerHelper::getAlignmentOptions(),
      '#empty_option' => '- Alignment -',
      '#attributes' => ['class' => ['ptm-global-alignment', 'd-none']],
      '#prefix'   => '<div class="ptm-global-alignment">',
      '#suffix'   => '</div>',
    ];

    // ------------------------------------------------------------
    // GRID WRAPPER FOR PARAGRAPH BUNDLES.
    // ------------------------------------------------------------
    $form['all_wrapper']['wrapper'] = [
      '#type' => 'container',
      '#tree' => TRUE,
      '#attributes' => ['class' => ['ptm-grid-wrapper']],
    ];

    // Load all paragraph types.
    $paragraph_types = ParagraphsType::loadMultiple();

    // Track if we’ve found at least one title field across all bundles.
    $has_any_title_fields = FALSE;

    // ============================================================
    // LOOP THROUGH EACH PARAGRAPH TYPE.
    // ============================================================
    foreach ($paragraph_types as $machine_name => $type) {

      // Auto-detect title fields for this paragraph type.
      $fields = $this->getTitleFields($machine_name);

      // Create section for this bundle.
      $form['all_wrapper']['wrapper'][$machine_name] = [
        '#type' => 'details',
        '#title' => $type->label(),
        '#open' => TRUE,
        '#tree' => TRUE,
        '#attributes' => ['class' => ['ptm-grid-item']],
      ];

      // ----------------------------------------------------------------
      // CASE 1: No title-like fields exist for this bundle.
      // ----------------------------------------------------------------
      if (empty($fields)) {
        $form['all_wrapper']['wrapper'][$machine_name]['no_fields'] = [
          '#markup' => '<div class="ptm-info-message">This paragraph type contains no title-like fields.</div>',
        ];
        continue;
      }

      // We have at least one title field in the whole site.
      $has_any_title_fields = TRUE;

      // ----------------------------------------------------------------
      // CASE 2: Show detected field machine names (human-friendly list).
      // ----------------------------------------------------------------
      $form['all_wrapper']['wrapper'][$machine_name]['fields_list'] = [
        '#markup' => '<div class="ptm-field-list"><strong>Detected fields:</strong> ' .
          implode(', ', array_keys($fields)) . '</div>',
      ];

      // ----------------------------------------------------------------
      // Build controls for each detected title field.
      // ----------------------------------------------------------------
      foreach ($fields as $field_name => $field_label) {

        $saved_row = $saved[$machine_name][$field_name] ?? NULL;
        $enabled   = $saved_row->enabled ?? 0;
        $alignment = $saved_row->alignment ?? 'center';

        // Enable/disable checkbox.
        $form['all_wrapper']['wrapper'][$machine_name][$field_name . '_enabled'] = [
          '#type' => 'checkbox',
          '#title' => "$field_label ($field_name)",
          '#default_value' => $enabled,
          '#attributes' => ['class' => ['ptm-title-checkbox']],
        ];

        // Alignment selector.
        $form['all_wrapper']['wrapper'][$machine_name][$field_name . '_alignment'] = [
          '#type' => 'select',
          '#title' => 'Alignment',
          '#options' => ParagraphsTitleManagerHelper::getAlignmentOptions(),
          '#default_value' => $alignment,
          '#states' => [
            'visible' => [
              ':input[name="all_wrapper[wrapper][' . $machine_name . '][' . $field_name . '_enabled]"]'
                => ['checked' => TRUE],
            ],
          ],
        ];

        // Insert default row if missing.
        $this->insertWhenTableEmpty($machine_name, $field_name, $enabled, $alignment);
      }
    }

    // ------------------------------------------------------------------
    // GLOBAL CASE: No title-like fields found in ANY paragraph bundle.
    // In that case, don’t show empty UI controls; just a clear message.
    // ------------------------------------------------------------------
    if (!$has_any_title_fields) {
      $form['all_wrapper']['wrapper'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['ptm-grid-wrapper']],
        'no_any_fields' => [
          '#markup' => '<div class="ptm-info-message ptm-info-global">
            No paragraph title fields were detected across any paragraph types.<br/>
            To use this module, add fields whose machine names contain 
            <code>title</code> (for example: <code>title</code>, <code>title_one</code>, <code>sub_title</code>) 
            to your paragraph types. They will appear here automatically.
          </div>',
        ],
      ];
    }

    // ------------------------------------------------------------
    // SUBMIT BUTTON.
    // ------------------------------------------------------------
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => 'Save settings',
      '#attributes' => [
        'onclick' => "return confirm('Apply global settings for selected items?');",
      ],
    ];

    // Attach library (JS + CSS).
    $form['#attached']['library'][] = 'paragraphs_title_manager/global';

    return $form;
  }

  /**
   * Inserts default rows safely if a bundle/field combination is missing.
   */
  public function insertWhenTableEmpty($machine_name, $field_name, $enabled, $alignment) {
    try {
      $connection = Database::getConnection();

      $exists = $connection->select('paragraphs_title_manager', 'ptm')
        ->fields('ptm', ['pid'])
        ->condition('pid', $machine_name)
        ->condition('field_name', $field_name)
        ->range(0, 1)
        ->execute()
        ->fetchField();

      if (!$exists) {
        $connection->insert('paragraphs_title_manager')
          ->fields([
            'pid' => $machine_name,
            'field_name' => $field_name,
            'enabled' => $enabled ?? 0,
            'alignment' => $alignment ?? 'center',
          ])
          ->execute();
      }
    }
    catch (\Exception $e) {
      \Drupal::messenger()->addError("Failed to create default settings for $machine_name → $field_name.");
    }
  }

  /**
   * Auto-detects all title-like fields for a paragraph bundle.
   *
   * Title fields here are any fields whose machine name contains "title",
   * for example: title, title_one, sub_title.
   */
  private function getTitleFields($bundle) {
    $fields = \Drupal::service('entity_field.manager')
      ->getFieldDefinitions('paragraph', $bundle);

    $result = [];

    foreach ($fields as $name => $field) {
      if (stripos($name, 'title') !== FALSE) {
        $result[$name] = $field->getLabel();
      }
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   * Save form settings safely.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    try {
      $connection = Database::getConnection();
      $paragraph_types = ParagraphsType::loadMultiple();

      foreach ($paragraph_types as $machine_name => $type) {

        $fields = $this->getTitleFields($machine_name);
        if (empty($fields)) {
          continue;
        }

        foreach ($fields as $field_name => $label) {

          $enabled = $form_state->getValue([
            'all_wrapper',
            'wrapper',
            $machine_name,
            $field_name . '_enabled',
          ]);

          $alignment = $form_state->getValue([
            'all_wrapper',
            'wrapper',
            $machine_name,
            $field_name . '_alignment',
          ]);

          // Check if row exists.
          $exists = $connection->select('paragraphs_title_manager', 'ptm')
            ->fields('ptm', ['pid'])
            ->condition('pid', $machine_name)
            ->condition('field_name', $field_name)
            ->range(0, 1)
            ->execute()
            ->fetchField();

          // Update or insert.
          if ($exists) {
            $connection->update('paragraphs_title_manager')
              ->fields([
                'enabled' => $enabled ?? 0,
                'alignment' => $alignment ?? 'center',
              ])
              ->condition('pid', $machine_name)
              ->condition('field_name', $field_name)
              ->execute();
          }
          else {
            $connection->insert('paragraphs_title_manager')
              ->fields([
                'pid' => $machine_name,
                'field_name' => $field_name,
                'enabled' => $enabled ?? 0,
                'alignment' => $alignment ?? 'center',
              ])
              ->execute();
          }
        }
      }

      \Drupal::messenger()->addMessage('Paragraph title alignment settings saved.');

    }
    catch (\Exception $e) {
      \Drupal::messenger()->addError('Failed to save Paragraph Title Manager settings.');
    }
  }

}
