<?php

namespace Drupal\parameters_ui\Form;

use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Render\Element;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Form for configuring one or multiple existing parameters.
 */
class ParametersConfigForm implements FormInterface, ContainerInjectionInterface {

  use DependencySerializationTrait;
  use AutowireTrait;
  use StringTranslationTrait;
  use MessengerTrait;

  /**
   * Involved parameters.
   *
   * @var \Drupal\parameters\Plugin\ParameterInterface[]
   */
  protected ?array $parameters = NULL;

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'parameters_config';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\parameters\Plugin\ParameterInterface[] $parameters */
    if (!isset($this->parameters)) {
      $this->parameters = $form_state->getBuildInfo()['args'][0];
    }
    $parameters = $this->parameters;

    $weight = 0;

    foreach ($parameters as $identifier => $parameter) {
      $weight += 100;
      $form[$identifier] = [
        '#parents' => [],
        '#weight' => $weight,
      ];
      $parameter_form_state = SubformState::createForSubform($form[$identifier], $form, $form_state);
      $form[$identifier] = $parameter->buildConfigurationForm($form[$identifier], $parameter_form_state);
      if (isset($form[$identifier]['value']['#title']) && 1 === count(Element::children($form[$identifier]))) {
        $form[$identifier]['value']['#title'] = $parameter->getLabel();
        $form[$identifier]['value']['#description'] = $parameter->getDescription();
      }
      else {
        $form[$identifier]['#type'] = 'details';
        $form[$identifier]['#open'] = TRUE;
        $form[$identifier]['#title'] = $parameter->getLabel();
        $form[$identifier]['description'] = [
          '#prefix' => '<p>',
          '#suffix' => '</p>',
          '#markup' => $parameter->getDescription(),
          '#weight' => -1000,
        ];
      }
    }

    $form['actions'] = ['#type' => 'actions', '#weight' => ($weight += 100)];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#button_type' => 'primary',
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\parameters\Plugin\ParameterInterface[] $parameters */
    $parameters = $this->parameters;

    foreach ($parameters as $identifier => $parameter) {
      $parameter_form_state = SubformState::createForSubform($form[$identifier], $form, $form_state);
      $parameter->validateConfigurationForm($form[$identifier], $parameter_form_state);
    }
    $collections = [];
    $collectionStorage = $this->entityTypeManager->getStorage('parameters_collection');
    foreach ($parameters as $identifier => $parameter) {
      [$id, $name] = explode(':', $identifier, 2);
      if (!isset($collections[$id])) {
        $collections[$id] = $collectionStorage->load($id);
      }
      if (!isset($collections[$id])) {
        $form_state->setError($form, $this->t('The changes cannot be saved due to missing collection "@id".', ['@id' => $id]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\parameters\Plugin\ParameterInterface[] $parameters */
    $parameters = $this->parameters;
    /** @var \Drupal\parameters\Entity\ParametersCollectionInterface[] $collections */
    $collections = [];
    $collectionStorage = $this->entityTypeManager->getStorage('parameters_collection');
    foreach ($parameters as $identifier => $parameter) {
      $parameter_form_state = SubformState::createForSubform($form[$identifier], $form, $form_state);
      $parameter->submitConfigurationForm($form[$identifier], $parameter_form_state);
      [$id] = explode(':', $identifier, 2);
      if (!isset($collections[$id])) {
        $collections[$id] = $collectionStorage->load($id);
      }
      /** @var \Drupal\parameters\Entity\ParametersCollectionInterface $collection */
      $collection = $collections[$id];

      $parameter_name = $parameter->getName();
      $parameters_array = $collection->get('parameters');
      $parameters_array[$parameter_name] = [
        'name' => $parameter_name,
        'type' => $parameter->getType(),
      ] + $parameter->getConfiguration();
      foreach ($parameter->getThirdPartyProviders() as $provider) {
        $parameters_array[$parameter_name]['third_party_settings'][$provider] = $parameter->getThirdPartySettings($provider);
      }
      $collection->setParameters($parameters_array);
    }

    foreach ($collections as $collection) {
      $collection->save();
    }

    $this->messenger()->addStatus($this->t('The configuration options have been saved.'));
  }

}
