<?php

namespace Drupal\parameters_ui\Form;

use Drupal\Core\Access\AccessibleInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\Core\Render\Markup;

/**
 * Form for assembling a form to configure one or multiple existing parameters.
 */
class ParametersConfigAssembleForm implements FormInterface, ContainerInjectionInterface {

  use DependencySerializationTrait;
  use AutowireTrait;
  use StringTranslationTrait;
  use MessengerTrait;

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'parameters_config_assemble';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['help'] = [
      '#prefix' => '<p>',
      '#suffix' => '</p>',
      '#markup' => $this->t('Select one or multiple of the following parameters that you want to assemble into one configuration form. Then copy the assembled link below and use it for accessing the assembled form.'),
    ];

    $form['table'] = [
      '#type' => 'table',
      '#header' => [
        'selected' => '',
        'collection' => $this->t('Collection'),
        'parameter' => $this->t('Parameter'),
        'label' => $this->t('Label'),
        'type' => $this->t('Type'),
      ],
    ];
    $collectionStorage = $this->entityTypeManager->getStorage('parameters_collection');
    $collections = $collectionStorage->loadMultiple();

    foreach ($collections as $collection) {
      foreach ($this->entityTypeManager->getStorage('parameters_collection')->loadMultiple() as $collection) {
        /** @var \Drupal\parameters\Entity\ParametersCollectionInterface $collection */
        if (!$collection->access('view')) {
          continue;
        }
        foreach ($collection->getParameters() as $parameter) {
          if (($parameter instanceof AccessibleInterface) && !$parameter->access('view')) {
            continue;
          }
          /** @var \Drupal\parameters\Plugin\ParameterInterface $parameter */
          $definition = $parameter->getPluginDefinition();
          $identifier = $collection->id() . ':' . $parameter->getName();
          $identifiers[$identifier] = $identifier;
          $form['table'][$identifier] = [
            'selected' => [
              '#type' => 'checkbox',
              '#title' => $this->t('Selected'),
              '#title_display' => 'invisible',
              '#parents' => ['selected', $identifier],
              '#return_value' => $identifier,
              '#default_value' => FALSE,
              '#attributes' => ['class' => ['param-checkbox']],
            ],
            'collection' => ['#markup' => $collection->id()],
            'parameter' => ['#markup' => $parameter->getName()],
            'label' => ['#markup' => $parameter->getLabel()],
            'type' => ['#markup' => $definition['label'] ?? $parameter->getType()],
          ];
        }
      }
    }

    $url = Url::fromRoute('parameters.config');
    $form['assembled'] = [
      '#prefix' => '<div id="assembled-link-wrapper" class="visually-hidden">',
      '#suffix' => '</div>',
      '#type' => 'textfield',
      '#title' => $this->t('Assembled link'),
      '#description' => $this->t('Copy this URL and share it with others or <a data-open-link-directly="assembled-link-wrapper" target="_blank" href="@url">open it directly.</a><br/>Please note: If you put this into other configuration, it may be advisable to use a relative URL by removing the host address.', [
        '@url' => $url->setAbsolute(FALSE)->toString(),
      ]),
      '#disabled' => TRUE,
      '#maxlength' => 100000,
      '#size' => 200,
      '#default_value' => $url->setAbsolute(TRUE)->toString(),
      '#attributes' => [
        'class' => ['assembled-input'],
        'data-base' => $url->setAbsolute(TRUE)->toString(),
      ],
    ];

    $js = <<<JS
(function(){
  var wrapper = document.getElementById('assembled-link-wrapper');
  var input = document.querySelector('input.assembled-input');
  var link = document.querySelector('a[data-open-link-directly="assembled-link-wrapper"]');
  if(!input){return;}
  var base = input.getAttribute('data-base') || input.value || '';
  function update(){
    var boxes = Array.prototype.slice.call(document.querySelectorAll('input.param-checkbox:checked'));
    var parts = boxes.map(function(cb){ return encodeURIComponent(cb.value); });
    if (parts.length){
      var url = base + '?parameters=' + parts.join('+');
      input.value = url;
      if(wrapper){ wrapper.classList.remove('visually-hidden'); }
      if(link){ link.setAttribute('href', url); }
    } else {
      input.value = base;
      if(wrapper){ wrapper.classList.add('visually-hidden'); }
      if(link){ link.setAttribute('href', base); }
    }
  }
  document.addEventListener('change', function(e){
    if(e.target && e.target.matches && e.target.matches('input.param-checkbox')){ update(); }
  }, true);
  update();
})();
JS;

    $form['js'] = ['#markup' => Markup::create('<script>' . $js . '</script>')];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['back_to_list'] = [
      '#type' => 'link',
      '#title' => $this->t('Back to list'),
      '#attributes' => ['class' => ['button', 'dialog-cancel']],
      '#url' => Url::fromRoute('entity.parameters_collection.collection'),
      '#weight' => 20,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {}

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {}

}
