<?php

namespace Drupal\parameters_ui\Controller;

use Drupal\Core\Access\AccessibleInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\parameters_ui\Form\ParametersConfigForm;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for configuring one or multiple existing parameters.
 */
class ParametersConfigController implements ContainerInjectionInterface {

  use AutowireTrait;

  public function __construct(
    protected FormBuilderInterface $formBuilder,
    protected EntityTypeManagerInterface $entityTypeManager
  ) {}

  /**
   * Returns a renderable array for a page for configuration.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return array
   *   Renderable array.
   */
  public function page(Request $request): array {
    $identifiers = $this->decodeIdentifiers($request->get('parameters') ?? []);
    $accessibleParameters = $this->loadAccessibleParameters($identifiers);
    return $this->formBuilder->getForm(ParametersConfigForm::class, $accessibleParameters);
  }

  /**
   * Custom access callback for this controller.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(Request $request): AccessResultInterface {
    $identifiers = $this->decodeIdentifiers($request->get('parameters') ?? []);
    $accessibleParameters = $this->loadAccessibleParameters($identifiers);
    return AccessResult::allowedIf(!empty($accessibleParameters))->setCacheMaxAge(0);
  }

  /**
   * Decodes the given identifiers coming from user input.
   *
   * @param string|array $identifiers
   *   The identifiers from user input.
   *
   * @return array
   *   All identifiers that could be successfully decoded.
   */
  protected function decodeIdentifiers(string|array $identifiers): array {
    $identifiersList = is_string($identifiers) ? preg_split('/\s*\+\s*/', strtr($identifiers, [' ' => '+']), -1, \PREG_SPLIT_NO_EMPTY) : (array) $identifiers;

    // Coerce to strings, drop non-strings, de-dup, and size-limit.
    $decoded = [];
    $seen = [];

    // Cap to prevent abuse.
    $maxItems = 200;

    foreach ($identifiersList as $raw) {
      if (!is_scalar($raw)) {
        continue;
      }

      $identifier = (string) $raw;
      if ($identifier === '' || isset($seen[$identifier])) {
        continue;
      }

      if (strlen($identifier) > 128) {
        // Cap item length.
        continue;
      }

      if (!str_contains($identifier, ':')) {
        continue;
      }

      [$id, $name] = explode(':', $identifier, 2);
      if ($id === '' || $name === '') {
        continue;
      }

      $decoded[$identifier] = [$id, $name];
      $seen[$identifier] = true;

      if (count($decoded) >= $maxItems) {
        break;
      }
    }

    return $decoded;
  }

  /**
   * Loads accessible parameter plugin instances.
   *
   * This acts in the following way: Once a parameter is not accessible, the
   * whole returned list is always empty.
   *
   * @param array $identifiers
   *   Decoded identifiers.
   *
   * @return \Drupal\parameters\Plugin\ParameterInterface[]
   *   Loaded parameter plugins.
   */
  protected function loadAccessibleParameters(array $identifiers): array {
    $collectionStorage = $this->entityTypeManager->getStorage('parameters_collection');
    $accessibleParameters = [];

    $collections = [];

    foreach ($identifiers as $identifier => $parts) {
      [$id, $name] = $parts;
      if (!isset($collections[$id])) {
        $collections[$id] = $collectionStorage->load($id);
      }
      if (!($collection = $collections[$id] ?? NULL)) {
        return [];
      }
      /** @var \Drupal\parameters\Entity\ParametersCollectionInterface $collection */
      if (!$collection->access('update')) {
        // Not granted to update the collection where the parameter is stored.
        return [];
      }
      if (!($parameter = $collection->getParameter($name))) {
        return [];
      }
      if (($parameter instanceof AccessibleInterface) && !$parameter->access('update', NULL, TRUE)->isAllowed()) {
        return [];
      }
      $accessibleParameters[$identifier] = $parameter;
    }

    return $accessibleParameters;
  }

}
