<?php

namespace Drupal\parameters\Plugin\Parameter;

use Drupal\Component\Render\MarkupInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\parameters\Attribute\Parameter;
use Drupal\parameters\Plugin\ParameterBase;

/**
 * Provides a color as parameter.
 */
#[Parameter(
  id: "color",
  label: new TranslatableMarkup("Color")
)]
class Color extends ParameterBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'value' => '#000000',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $default_value = (string) ($this->configuration['value'] ?? '#000000');

    $form['use_text_input'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use text input instead of color picker'),
      '#default_value' => FALSE,
      '#description' => $this->t('When enabled, a simple text field is shown for entering a hex color value (e.g., #1a2b3c).'),
    ];

    // Native color picker (visible when toggle is off).
    $form['value'] = [
      '#type' => 'color',
      '#title' => $this->t('Color'),
      '#default_value' => $default_value,
      '#states' => [
        'visible' => [
          ':input[name="use_text_input"]' => ['checked' => FALSE],
        ],
      ],
    ] + ($form['value'] ?? []);

    // Plain text input for hex (visible when toggle is on).
    $form['value_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Hex color'),
      '#default_value' => $default_value,
      '#size' => 10,
      '#maxlength' => 12,
      '#placeholder' => '#1a2b3c',
      '#description' => $this->t('Enter a hex color like #1a2b3c or #fff.'),
      '#states' => [
        'visible' => [
          ':input[name="use_text_input"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validate() {
    $value = $this->configuration['value'] ?? '';
    // Accept #RGB or #RRGGBB.
    if (!preg_match('/^#(?:[0-9a-fA-F]{3}){1,2}$/', $value)) {
      return $this->t('Please set a valid hex color (e.g. #1a2b3c or #fff).');
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // Decide which input to read.
    $use_text = (bool) $form_state->getValue('use_text_input');
    if (!$use_text) {
      parent::submitConfigurationForm($form, $form_state);
      return;
    }

    $raw = $use_text ? (string) $form_state->getValue('value_text') : (string) $form_state->getValue('value');

    // Normalize: ensure leading '#', lower-case.
    $raw = trim($raw);
    if ($raw !== '' && $raw[0] !== '#') {
      $raw = '#' . $raw;
    }
    $this->configuration['value'] = strtolower($raw);
  }

  /**
   * {@inheritdoc}
   */
  public function getPreview(): MarkupInterface {
    $hex = $this->configuration['value'] ?? '#000000';
    $swatch = '<span style="position:relative;top:-2px;display:inline-block;width:1em;height:1em;vertical-align:middle;margin-right:.4em;border:1px solid #ccc;background:' . htmlspecialchars($hex, ENT_QUOTES, 'UTF-8') . ';"></span>';
    return Markup::create($swatch . htmlspecialchars($hex, ENT_QUOTES, 'UTF-8'));
  }

}
