<?php

namespace Drupal\parameters\Template;

use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Render\Markup;
use Drupal\Core\Render\RenderableInterface;
use Drupal\parameters\Entity\ParametersCollectionInterface;
use Drupal\parameters\Parameter;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

/**
 * Twig extensions provided by the Parameters module.
 */
class ParametersTwigExtension extends AbstractExtension {

  /**
   * {@inheritdoc}
   */
  public function getFunctions() {
    return [
      new TwigFunction('p', $this->getRenderableParameter(...)),
    ];
  }

  /**
   * Get a parameter in a renderable format.
   *
   * Optionally set "value" as additional argument for returning the value
   * instead of a renderable format. This may be more efficient for conditions.
   *
   * @param string $name
   *   The machine name of the parameter.
   * @param mixed ...$context
   *   Additional arguments used as according context. This is mostly an entity.
   *
   * @return mixed
   *   The parameter as renderable array, or the value directly in case "value"
   *   was given as additional argument.
   */
  public function getRenderableParameter($name, ...$context): mixed {
    assert(is_string($name), 'Name argument must be a string.');

    if (in_array('value', $context, TRUE)) {
      return Parameter::value($name, ...$context);
    }

    $parameter = Parameter::get($name, ...$context);

    $build = $parameter instanceof RenderableInterface ? $parameter->toRenderable() : [
      '#markup' => Markup::create($parameter->getProcessedData()->getString()),
    ];

    $metadata = BubbleableMetadata::createFromRenderArray($build);
    $metadata->addCacheTags(\Drupal::entityTypeManager()->getDefinition(ParametersCollectionInterface::ENTITY_TYPE_ID)->getListCacheTags());
    if ($parameter instanceof CacheableDependencyInterface) {
      $metadata->addCacheableDependency($parameter);
    }
    foreach ($context as $item) {
      if ($item instanceof CacheableDependencyInterface) {
        $metadata->addCacheableDependency($item);
      }
    }
    $metadata->applyTo($build);

    return $build;
  }

}
