<?php

namespace Drupal\parameters\Plugin\Derivative\Parameter;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Parameter plugin deriver for bundles.
 *
 * @see \Drupal\parameters\Plugin\Parameter\Bundles
 */
class BundlesDeriver implements ContainerDeriverInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Creates a new ContentDerivativeBase object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    $instance = new static(
      $container->get('entity_type.manager')
    );
    $instance->setStringTranslation($container->get('string_translation'));
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinition($derivative_id, $base_plugin_definition) {
    $entity_type_id = $derivative_id;
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id, FALSE);

    if (!$entity_type || !($entity_type->entityClassImplements(ContentEntityInterface::class)) || (NULL === $entity_type->getBundleEntityType())) {
      return NULL;
    }

    $bundle_entity_type = $this->entityTypeManager->getDefinition($entity_type->getBundleEntityType());

    $definition = [
      'label' => new TranslatableMarkup('Selection of @label', ['@label' => $bundle_entity_type->getPluralLabel()]),
      'entity_type' => $entity_type_id,
    ] + $base_plugin_definition;

    return $definition;
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {
    $derivatives = [];

    foreach (array_keys($this->entityTypeManager->getDefinitions()) as $entity_type_id) {
      if ($derivative_definition = $this->getDerivativeDefinition($entity_type_id, $base_plugin_definition)) {
        $derivatives[$entity_type_id] = $derivative_definition;
      }
    }

    return $derivatives;
  }

}
