<?php

namespace Drupal\parameters\Plugin;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\parameters\Attribute\Parameter;
use Drupal\parameters\Plugin\Parameter\NullObject;

/**
 * The manager for parameter plugins.
 */
class ParameterManager extends DefaultPluginManager {

  /**
   * The service name of this class.
   *
   * @var string
   */
  const SERVICE_NAME = 'plugin.manager.parameter';

  /**
   * Get the service instance of this class.
   *
   * @return \Drupal\parameters\Plugin\ParameterManager
   *   The service instance.
   */
  public static function get(): PluginManagerInterface {
    return \Drupal::service(self::SERVICE_NAME);
  }

  /**
   * Runtime cache of lazy-loaded definitions.
   *
   * @var array
   */
  protected ?array $lazyDefinitions = NULL;

  /**
   * The cache key used to identify cached lazy-loaded definitions.
   *
   * @var string
   */
  protected string $cacheKeyLazy = 'parameter_plugins:lazy';

  /**
   * The ParameterManager constructor.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct('Plugin/Parameter', $namespaces, $module_handler, 'Drupal\parameters\Plugin\ParameterInterface', Parameter::class, 'Drupal\parameters\Annotation\Parameter');
    $this->alterInfo('parameter_info');
    $this->setCacheBackend($cache_backend, 'parameter_plugins', ['parameter_plugins']);
  }

  /**
   * Creates a pre-configured instance of a plugin.
   *
   * @param string $plugin_id
   *   The ID of the plugin being instantiated.
   * @param array $configuration
   *   An array of configuration relevant to the plugin instance.
   *
   * @return \Drupal\parameters\Plugin\ParameterInterface
   *   A fully configured plugin instance.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   *   If the instance cannot be created, such as if the ID is invalid.
   */
  public function createInstance($plugin_id, array $configuration = []) {
    return parent::createInstance($plugin_id, $configuration);
  }

  /**
   * {@inheritdoc}
   */
  protected function findDefinitions() {
    $definitions = parent::findDefinitions();
    uasort($definitions, function($a, $b) {
      $a_weight = $a['weight'] ?? 0;
      $b_weight = $b['weight'] ?? 0;
      return $a_weight !== $b_weight ? $a_weight - $b_weight : strnatcmp((string) ($a['label'] ?? ''), (string) ($b['label'] ?? ''));
    });
    return $definitions;
  }

  /**
   * Get derivative definitions for the given base plugin ID.
   *
   * @param string $base_plugin_id
   *   The base plugin ID.
   *
   * @return array
   *   Derivative plugin definitions.
   */
  public function getDerivativeDefinitionsForBasePluginId(string $base_plugin_id): array {
    $definition = $this->getDefinition($base_plugin_id);
    if (!empty($definition['lazyDeriver'])) {
      $definition['deriver'] = $definition['lazyDeriver'];
    }
    $definitions = [$base_plugin_id => $definition];
    $derivatives = \Closure::fromCallable(function(array $definitions) {
      /** @var \Drupal\Core\Plugin\Discovery\ContainerDerivativeDiscoveryDecorator $this */
      return $this->getDerivatives($definitions);
    })->call($this->getDiscovery(), $definitions);
    return $derivatives;
  }

  /**
   * {@inheritdoc}
   */
  protected function doGetDefinition(array $definitions, $plugin_id, $exception_on_invalid) {
    $this->beforeDoGetDefinition($definitions, $plugin_id, $exception_on_invalid);
    return parent::doGetDefinition($definitions, $plugin_id, $exception_on_invalid);
  }

  /**
   * Act on given definitions before finally getting a plugin definition.
   *
   * @param array &$definitions
   *   Existing plugin definitions.
   * @param string $plugin_id
   *   Plugin ID.
   * @param bool $exception_on_invalid
   *   Whether to throw an exception when invalid.
   */
  protected function beforeDoGetDefinition(array &$definitions, $plugin_id, $exception_on_invalid): void {
    if (isset($definitions[$plugin_id])) {
      return;
    }

    if (!isset($this->lazyDefinitions)) {
      if ($cached = $this->cacheGet($this->cacheKeyLazy)) {
        $this->lazyDefinitions = $cached->data;
      }
      else {
        $this->lazyDefinitions = [];
      }
    }

    if (isset($this->lazyDefinitions[$plugin_id])) {
      $definitions[$plugin_id] = $this->lazyDefinitions[$plugin_id];
      return;
    }

    if (($base_plugin_id = strstr($plugin_id, ':', TRUE))
    && ($base_plugin_definition = $definitions[$base_plugin_id] ?? NULL)
    && isset($base_plugin_definition['lazyDeriver'])) {
      $definition = \Closure::fromCallable(function($plugin_id, $base_plugin_definition) {
        /** @var \Drupal\Core\Plugin\Discovery\ContainerDerivativeDiscoveryDecorator $this */
        $base_plugin_definition['deriver'] = $base_plugin_definition['lazyDeriver'] ?? ($base_plugin_definition['deriver'] ?? NULL);
        [$base_plugin_id, $derivative_id] = $this->decodePluginId($plugin_id);

        // Unset before and afterwards is intentional. The lazy deriver may not
        // interfere with a standard deriver.
        unset($this->derivers[$base_plugin_id]);
        $deriver = $this->getDeriver($base_plugin_id, $base_plugin_definition, $plugin_id);
        unset($this->derivers[$base_plugin_id]);

        $plugin_definition = $deriver ? $deriver->getDerivativeDefinition($derivative_id, $base_plugin_definition) : NULL;

        return $plugin_definition;
      })->call($this->getDiscovery(), $plugin_id, $base_plugin_definition);

      if ($definition) {
        $definitions[$plugin_id] = $definition;
        $this->lazyDefinitions[$plugin_id] = $definition;
        $this->cacheSet(cid: $this->cacheKeyLazy, data: $this->lazyDefinitions, tags: $this->cacheTags);
      }

      return;
    }

    // The "content" plugin got moved into "parameters_content" sub-module,
    // which is required to be installed for using this plugin. This case is
    // usually covered automatically by an update hook. As long as that has not
    // been run yet, log an error and replace the returning plugin with a
    // NullObject class definition.
    // @todo Remove this part on a major version upgrade such as 2.x.
    if (!isset($definitions[$plugin_id]) && str_starts_with($plugin_id, 'content')) {
      \Drupal::logger('parameters')->error("A \"Content\" parameter is in use but cannot be loaded due to an invalid state of this site. Please run database updates and make sure the \"parameters_content\" module is installed.");
      $definitions[$plugin_id] = [
        'id' => isset($base_plugin_id) ? $base_plugin_id : $plugin_id,
        'label' => 'Content (<b>broken</b> - see logged errors)',
        // Not using the legacy Content class. It will be removed on a version
        // upgrade and would cause class-not-found fatals due to cached
        // definitions. Therefore rely on the NullObject here, which will also
        // be available after a major upgrade.
        'class' => NullObject::class,
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function useCaches($use_caches = FALSE) {
    parent::useCaches($use_caches);
    if (!$use_caches) {
      $this->lazyDefinitions = NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function clearCachedDefinitions() {
    parent::clearCachedDefinitions();
    $this->lazyDefinitions = NULL;
    if (!$this->cacheTags) {
      $this->cacheBackend->delete($this->cacheKeyLazy);
    }
  }

}
