<?php

declare(strict_types=1);

namespace Drupal\parli_protect\EventSubscriber;

use Drupal\Core\Routing\LocalRedirectResponse;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\parli_protect\ParliProtectManager;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber to react to UK Parliament users.
 */
class ParliProtectSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a ParliProtectSubscriber object.
   */
  public function __construct(
    private readonly AccountProxyInterface $currentUser,
    private readonly ParliProtectManager $parliProtectManager,
  ) {}

  /**
   * Redirect UK Parliament users to OSA Verification form.
   */
  public function onKernelRequest(RequestEvent $event): void {
    // Only return a response for a master request.
    if (!$event->isMainRequest()) {
      return;
    }

    // Allow for bypass.
    if ($this->currentUser->hasPermission('bypass parli_protect protection')) {
      return;
    }

    // Don't redirect if we're already on the verification page.
    $redirect_url = Url::fromRoute('parli_protect.protection')->toString();
    if ($event->getRequest()->getPathInfo() == $redirect_url) {
      return;
    }

    $ip = $event->getRequest()->getClientIp();
    if ($this->parliProtectManager->isParliament($ip)) {
      $event->setResponse(new LocalRedirectResponse($redirect_url));
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::REQUEST => ['onKernelRequest', 28],
    ];
  }

}
