<?php

declare(strict_types=1);

namespace Drupal\parli_protect\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\filter\Element\ProcessedText;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a Parliament Protect form.
 */
class VerificationForm extends FormBase {

  /**
   * The Parliament Protect configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $parliProtectConfig;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    private readonly LanguageManagerInterface $languageManager,
  ) {
    $this->parliProtectConfig = $this->config('parli_protect.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('language_manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'parli_protect_verification';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    $form['intro'] = [
      '#type' => 'processed_text',
      '#text' => $this->parliProtectConfig->get('form_intro.value'),
      '#format' => $this->parliProtectConfig->get('form_intro.format'),
    ];

    $form['title'] = [
      '#type' => 'select',
      '#title' => $this->t('Title'),
      '#options' => [
        'Mx',
        'Ms',
        'Miss',
        'Mr',
        'Dr',
        'Sir',
        'Lord',
        'Baroness',
        'Duke',
        'Marquess',
        'Earl',
        'Viscount',
        'Baron',
        'Master',
        'Mistress',
        'Daddy',
        'Brat',
      ],
      '#required' => TRUE,
    ];

    $form['first_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('First name(s)'),
      '#required' => TRUE,
    ];

    $form['last_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Last name'),
      '#required' => TRUE,
      '#description' => $this->t('Please enter double-barelled last names in the same field.'),
    ];

    $form['pronouns'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pronouns'),
      '#required' => TRUE,
    ];

    $form['gender'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Gender(s)'),
      '#required' => TRUE,
    ];

    $form['dob'] = [
      '#type' => 'datelist',
      '#title' => $this->t('Date of birth'),
      '#required' => TRUE,
    ];

    $form['born'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Where were you born?'),
      '#required' => TRUE,
      '#description' => $this->t('Please enter town/city, country, planet, and galaxy.'),
    ];

    $form['ethnicity'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Ethnicity'),
      '#required' => TRUE,
    ];

    $form['eye_colour'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Eye colour'),
      '#required' => TRUE,
    ];

    $form['hair_colour'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Hair colour'),
      '#required' => TRUE,
    ];

    $form['teeth'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Are those your original teeth?'),
      '#open' => TRUE,
    ];

    $form['teeth']['no'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('No'),
      '#required' => TRUE,
    ];

    $form['employer'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Employer'),
      '#required' => TRUE,
      '#description' => $this->t('Please list all the other people paying you to act in their interests outside of UK taxpayers.'),
    ];

    $form['email_work'] = [
      '#type' => 'email',
      '#title' => $this->t('Email (work)'),
      '#required' => TRUE,
    ];

    $form['email_personal'] = [
      '#type' => 'email',
      '#title' => $this->t('Email (personal)'),
      '#required' => TRUE,
    ];

    $form['email_alt'] = [
      '#type' => 'email',
      '#title' => $this->t('Email (alt)'),
      '#required' => TRUE,
    ];

    $form['landline'] = [
      '#type' => 'tel',
      '#title' => $this->t('Landline number'),
      '#required' => TRUE,
    ];

    $form['mobile'] = [
      '#type' => 'tel',
      '#title' => $this->t('Mobile number'),
      '#required' => TRUE,
    ];

    $form['burner'] = [
      '#type' => 'tel',
      '#title' => $this->t('Burner number'),
      '#required' => TRUE,
    ];

    $form['address_intro'] = [
      '#type' => 'item',
      '#title' => 'Addresses',
      '#markup' => $this->t('<p>There are three addresses available on the form. While we appreciate most members of UK Parliament will have more, if you run out of space only list the ones where you spend most of your time, most likely <em>outside</em> of your constituency.</p>'),
    ];

    $form['address_1'] = [
      '#type' => 'details',
      '#title' => $this->t('Address 1'),
      '#open' => TRUE,
    ];

    $form['address_1']['house_name_number'] = [
      '#type' => 'textfield',
      '#title' => $this->t('House Name or Number'),
      '#required' => TRUE,
    ];

    $form['address_1']['street'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Street'),
      '#required' => TRUE,
    ];

    $form['address_1']['additional'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Additional'),
      '#description' => $this->t('Flat or office suite number, penthouse, hotel room, poolhouse, etc.'),
      '#required' => TRUE,
    ];

    $form['address_1']['city'] = [
      '#type' => 'textfield',
      '#title' => $this->t('City/Town/Village'),
      '#required' => TRUE,
    ];

    $form['address_1']['county'] = [
      '#type' => 'textfield',
      '#title' => $this->t('County'),
      '#required' => TRUE,
    ];

    $form['address_1']['postcode'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Postcode'),
      '#required' => TRUE,
    ];

    $form['address_2'] = [
      '#type' => 'details',
      '#title' => $this->t('Address 2'),
      '#open' => TRUE,
    ];

    $form['address_2']['house_name_number'] = [
      '#type' => 'textfield',
      '#title' => $this->t('House Name or Number'),
      '#required' => TRUE,
    ];

    $form['address_2']['street'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Street'),
      '#required' => TRUE,
    ];

    $form['address_2']['additional'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Additional'),
      '#description' => $this->t('Flat or office suite number, penthouse, hotel room, poolhouse, etc.'),
      '#required' => TRUE,
    ];

    $form['address_2']['city'] = [
      '#type' => 'textfield',
      '#title' => $this->t('City/Town/Village'),
      '#required' => TRUE,
    ];

    $form['address_2']['county'] = [
      '#type' => 'textfield',
      '#title' => $this->t('County'),
      '#required' => TRUE,
    ];

    $form['address_2']['postcode'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Postcode'),
      '#required' => TRUE,
    ];

    $form['address_3'] = [
      '#type' => 'details',
      '#title' => $this->t('Address 3'),
      '#open' => TRUE,
    ];

    $form['address_3']['house_name_number'] = [
      '#type' => 'textfield',
      '#title' => $this->t('House Name or Number'),
      '#required' => TRUE,
    ];

    $form['address_3']['street'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Street'),
      '#required' => TRUE,
    ];

    $form['address_3']['additional'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Additional'),
      '#description' => $this->t('Flat or office suite number, penthouse, hotel room, poolhouse, etc.'),
      '#required' => TRUE,
    ];

    $form['address_3']['city'] = [
      '#type' => 'textfield',
      '#title' => $this->t('City/Town/Village'),
      '#required' => TRUE,
    ];

    $form['address_3']['county'] = [
      '#type' => 'textfield',
      '#title' => $this->t('County'),
      '#required' => TRUE,
    ];

    $form['address_3']['postcode'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Postcode'),
      '#required' => TRUE,
    ];

    $form['pet'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pet\'s name'),
      '#required' => TRUE,
      '#description' => $this->t('If no pet enter NONE.'),
    ];

    $form['mum'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Mother\'s maiden name'),
      '#required' => TRUE,
      '#description' => $this->t('If no mother enter NONE.'),
    ];

    $form['band'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Favourite band'),
      '#required' => TRUE,
      '#description' => $this->t('"The Smiths" is not allowed.'),
    ];

    $form['food'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Favourite food'),
      '#required' => TRUE,
    ];

    $form['current_account_no'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Current account number'),
      '#required' => TRUE,
    ];

    $form['current_account_sort'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Current account sort code'),
      '#required' => TRUE,
    ];

    $form['debit_card_no'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Debit Card Number'),
      '#required' => TRUE,
    ];

    $form['debit_card_cvv'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Debit Card CVV'),
      '#required' => TRUE,
    ];

    $form['debit_card_pin'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Debit Card PIN'),
      '#required' => TRUE,
    ];

    $form['spain'] = [
      '#type' => 'datelist',
      '#title' => $this->t('When did you last go to Spain?'),
      '#required' => TRUE,
    ];

    $form['football'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Football club you pretend to support'),
      '#required' => TRUE,
    ];

    $form['youth_org'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Youth organisation you pretend to support'),
      '#required' => TRUE,
    ];

    $form['lgbt'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('You talk a lot about supporting LGBT+ people. Do you actually support them?'),
      '#open' => TRUE,
    ];

    $form['lgbt']['no'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('No'),
      '#required' => TRUE,
    ];

    $form['rowling'] = [
      '#type' => 'datelist',
      '#title' => $this->t('When did you last have lunch with JK Rowling?'),
      '#required' => TRUE,
    ];

    $form['shoe_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Shoe size'),
      '#required' => TRUE,
    ];

    $form['time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('What time is it?'),
      '#required' => TRUE,
    ];

    $form['servant'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Favourite servants name'),
      '#description' => $this->t('If you cannot remember their name enter JEEVES.'),
      '#required' => TRUE,
    ];

    $form['x'] = [
      '#type' => 'details',
      '#title' => $this->t('X (Social media network formerly known as Twitter which of course you are still on despite the actions of the current owner)'),
      '#open' => TRUE,
    ];

    $form['x']['username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username'),
      '#required' => TRUE,
    ];

    $form['x']['pass'] = [
      '#type' => 'password_confirm',
      '#title' => $this->t('Password'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'file',
      '#title' => $this->t('Upload your driving license or passport'),
      '#required' => TRUE,
    ];

    $form['selfie'] = [
      '#type' => 'file',
      '#title' => $this->t('Upload a selfie'),
      '#required' => TRUE,
    ];

    $form['message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('All other relevant or irrelevant personal information:'),
      '#required' => TRUE,
    ];

    $form['agree'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('I agree'),
      '#required' => TRUE,
    ];

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Send'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $band = $form_state->getValue('band');
    if ($band && mb_strtolower($band) == 'the smiths') {
      $form_state->setErrorByName(
        'band',
        $this->t("We told you The Smiths wasn't allowed."),
      );
    }
    if ($band && mb_strtolower($band) == 'oasis') {
      $form_state->setErrorByName(
        'band',
        $this->t("The correct answer is Blur."),
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $message = ProcessedText::preRenderText([
      '#text' => $this->parliProtectConfig->get('form_confirmation.value'),
      '#format' => $this->parliProtectConfig->get('form_confirmation.format'),
      '#langcode' => $this->languageManager->getCurrentLanguage()->getId(),
      '#filter_types_to_skip' => [],
    ]);
    $this->messenger()->addStatus($message);
  }

}
