<?php

declare(strict_types=1);

namespace Drupal\parli_protect;

use Drupal\Core\Path\PathMatcherInterface;
use Drupal\Core\Routing\LocalRedirectResponse;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\parli_protect\ParliProtectManager;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\HttpKernelInterface;

/**
 * Middleware to serve up trouble to UK Parliament.
 */
final class ParliProtectMiddleware implements HttpKernelInterface {

  /**
   * Constructs a ParliProtectMiddleware object.
   */
  public function __construct(
    private readonly HttpKernelInterface $httpKernel,
    private readonly AccountProxyInterface $currentUser,
    private readonly ParliProtectManager $parliProtectManager,
    private readonly PathMatcherInterface $pathMatcher,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function handle(Request $request, $type = self::MAIN_REQUEST, $catch = TRUE): Response {
    $ip = $request->getClientIp();
    $exceptions = $this->pathExceptions();
    $exception_string = implode("\n", $exceptions);
    if (!$this->pathMatcher->matchPath($request->getPathInfo(), $exception_string) && $this->parliProtectManager->isParliament($ip)) {
      return new LocalRedirectResponse('/parliament-protect');
    }
    return $this->httpKernel->handle($request, $type, $catch);
  }

  /**
   * Exceptions to the rule.
   */
  public function pathExceptions() {
    return [
      '/parliament-protect',
      '/admin/config/system/parli-protect',
      '/sites/default/files/css/*',
      '/sites/default/files/js/*',
    ];
  }

}
