<?php

declare(strict_types=1);

namespace Drupal\parli_protect\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\State\StateInterface;
use Drupal\parli_protect\ParliProtectManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Returns responses for Parliament Protect routes.
 */
class ProtectionController extends ControllerBase {

  /**
   * The Parliament Protect configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $parliProtectConfig;

  /**
   * The controller constructor.
   */
  public function __construct(
    private readonly ParliProtectManager $parliProtectManager,
    private readonly RequestStack $requestStack,
    private readonly StateInterface $state,
  ) {
    $this->parliProtectConfig = $this->config('parli_protect.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('parli_protect.manager'),
      $container->get('request_stack'),
      $container->get('state'),
    );
  }

  /**
   * Builds the response.
   */
  public function __invoke(): array {
    $protection = $this->parliProtectConfig->get('protection');

    switch ($protection) {
      case 'form':
        $build['form'] = $this->formBuilder()->getForm('\Drupal\parli_protect\Form\VerificationForm');
        break;

      case 'message':
        $build['content'] = [
          '#type' => 'processed_text',
          '#text' => $this->parliProtectConfig->get('message.value'),
          '#format' => $this->parliProtectConfig->get('message.format'),
          '#cache' => [
            'tags' => [
              'parli_protect',
            ]
          ]
        ];
        break;
    }

    return $build;
  }

  /**
   * Title callback for the protection route.
   */
  public function title() {
    return $this->parliProtectConfig->get('title');
  }

}
