<?php

declare(strict_types=1);

namespace Drupal\parli_protect\Form;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Parliament Protect settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->state = $container->get('state');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'parli_protect_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['parli_protect.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['stats'] = [
      '#type' => 'item',
      '#title' => $this->t('Statistics'),
      '#markup' => $this->formatPlural($this->state->get('parli_protect_counter'), '@count request from UK Parliament has been blocked.', '@count requests from UK Parliament have been blocked.'),
    ];

    $form['ip_ranges'] = [
      '#type' => 'textarea',
      '#title' => $this->t('IP Ranges or Addresses'),
      '#description' => $this->t('Enter IP ranges in CIDR format or individual IP addresses. Each entry should be on a new line.'),
      '#default_value' => $this->config('parli_protect.settings')->get('ip_ranges'),
      '#required' => TRUE,
    ];

    $form['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Title'),
      '#description' => $this->t('Title to be shown on the protection page.'),
      '#default_value' => $this->config('parli_protect.settings')->get('title'),
      '#required' => TRUE,
    ];

    $form['protection'] = [
      '#type' => 'radios',
      '#title' => $this->t('Protection'),
      '#description' => $this->t('Determines what is shown to parliamentary visitors.'),
      '#options' => [
        'form' => $this->t('Form'),
        'message' => $this->t('Message'),
      ],
      '#default_value' => $this->config('parli_protect.settings')->get('protection'),
      '#required' => TRUE,
    ];

    $form['form_intro'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Form introduction'),
      '#description' => $this->t('Displayed above the form when "Form" is selected as the protection method.'),
      '#format' => $this->config('parli_protect.settings')->get('form_intro.format'),
      '#default_value' => $this->config('parli_protect.settings')->get('form_intro.value'),
      '#states' => [
        'visible' => [
          ':input[name="protection"]' => ['value' => 'form'],
        ],
      ],
    ];

    $form['form_confirmation'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Form confirmation message'),
      '#description' => $this->t('Message shown after the form is submitted.'),
      '#format' => $this->config('parli_protect.settings')->get('form_confirmation.format'),
      '#default_value' => $this->config('parli_protect.settings')->get('form_confirmation.value'),
      '#states' => [
        'visible' => [
          ':input[name="protection"]' => ['value' => 'form'],
        ],
      ],
    ];

    $form['message'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Message'),
      '#description' => $this->t('Displayed when "Message" is selected as the protection method.'),
      '#format' => $this->config('parli_protect.settings')->get('message.format'),
      '#default_value' => $this->config('parli_protect.settings')->get('message.value'),
      '#states' => [
        'visible' => [
          ':input[name="protection"]' => ['value' => 'message'],
        ],
      ],
    ];

    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced'),
    ];

    $form['advanced']['debug_enable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable debugging'),
      '#description' => $this->t('When not on this page, enabling debugging will cause you to be redirected, too.'),
      '#default_value' => $this->config('parli_protect.settings')->get('debug_enable'),
    ];

    $form['advanced']['debug_ip_address'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Debug IP Address'),
      '#description' => $this->t('An IP address to use for debugging.'),
      '#default_value' => $this->config('parli_protect.settings')->get('debug_ip_address'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    Cache::invalidateTags(['parli_protect']);
    $this->config('parli_protect.settings')
      ->set('ip_ranges', $form_state->getValue('ip_ranges'))
      ->set('title', $form_state->getValue('title'))
      ->set('protection', $form_state->getValue('protection'))
      ->set('form_intro', $form_state->getValue('form_intro'))
      ->set('form_confirmation', $form_state->getValue('form_confirmation'))
      ->set('message', $form_state->getValue('message'))
      ->set('debug_enable', $form_state->getValue('debug_enable'))
      ->set('debug_ip_address', $form_state->getValue('debug_ip_address'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
