<?php

declare(strict_types=1);

namespace Drupal\partytown\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure PartyTown settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  const PARTYTOWN_LOG_TYPES = [
    'logCalls',
    'logGetters',
    'logSetters',
    'logImageRequests',
    'logScriptExecution',
    'logSendBeaconRequests',
    'logStackTraces',
  ];

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'partytown_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['partytown.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    $form['enable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Partytown'),
      '#default_value' => $this->config('partytown.settings')->get('enable'),
      '#description' => $this->t('Enable the Partytown library and attach it to the page.'),
    ];

    $form['lib'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Lib Path'),
      '#default_value' => $this->config('partytown.settings')->get('lib'),
      '#description' => $this->t('The path to the partytown lib directory,
        with leading and trailing slashes.
       Default is <code>/libraries/partytown/lib/</code>'),
    ];

    $form['forwards'] = [
      '#type' => 'details',
      '#title' => 'Forwards',
      '#description' => $this->t('A list of function calls
        on the main thread to forward to the web worker.<br>
        See <a href="https://partytown.qwik.dev/forwarding-events" target="_blank">
        Forwarding Events and Triggers</a> for more info.'),
    ];

    $form['forwards']['actions'] = [
      '#type' => 'actions',
    ];

    $form['forwards']['actions']['add'] = [
      '#type' => 'submit',
      '#value' => $this->t('+ Add'),
      '#weight' => 100,
      '#submit' => ['::addForward'],
      '#ajax' => [
        'callback' => '::updateForwardAjaxCallback',
        'disable-refocus' => FALSE,
        'event' => 'click',
        'wrapper' => 'forwards-wrapper-ajax',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Adding forward...'),
        ],
      ],
    ];

    $form['forwards']['forwards_wrapper'] = [
      '#type' => 'fieldset',
      '#prefix' => '<div id="forwards-wrapper-ajax">',
      '#suffix' => '</div>',
    ];

    $forward_count = $form_state->get('forward_count');
    $forwards = $this->config('partytown.settings')->get('forward') ?: [];
    if (empty($forward_count)) {
      $forward_count = count($forwards) ?: 1;
      $form_state->set('forward_count', $forward_count);
    }

    // Render the current values, if any.
    for ($i = 0; $i < $forward_count; $i++) {
      $default_value = $form_state->getValue(['forwards', 'forwards_wrapper', 'function_' . $i])
        ?? ($forwards[$i]['function'] ?? '');
      $form['forwards']['forwards_wrapper']['function_' . $i] = [
        '#type' => 'textfield',
        '#title' => 'Function',
        '#description' => $this->t('The name of a function call to forward<br>
          ex. <code>dataLayer.push</code>'),
        '#default_value' => $default_value,
      ];

      $default_value = $form_state->getValue(['forwards', 'forwards_wrapper', 'preserve_behavior_' . $i])
        ?? ($forwards[$i]['preserve_behavior'] ?? '');
      $form['forwards']['forwards_wrapper']['preserve_behavior_' . $i] = [
        '#type' => 'checkbox',
        '#title' => 'Preserve Behavior',
        '#description' => $this->t('When preserveBehavior is set to true,
          the original function\'s behavior on the main thread is maintained,
          while also forwarding the calls to partytown.<br>
          See <a href="https://partytown.qwik.dev/forwarding-events/#preservebehavior" target="_blank">
          Forwarding Events</a> for more information.'),
        '#suffix' => '<hr>',
        '#default_value' => $default_value,
      ];
    }

    $form['load_scripts_on_main_thread'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Load Scripts On Main Thread'),
      '#description' => $this->t('An array of strings or regular expressions
        (RegExp) used to filter out which script are executed via Partytown
        and the main thread.<br>
        <br>
        An example is as follows:<br>
        <code>https://test.com/analytics.js</code><br>
        <code>inline-script-id</code><br>
        <code>/regex-matched-script\.js/</code><br>
        <br>
        Separate each rule with a newline.'),
      '#default_value' => $this->config('partytown.settings')->get('load_scripts_on_main_thread'),
    ];

    $form['fallback_timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Fallback Timeout'),
      '#description' => $this->t('A timeout in ms until Partytown initialization is
        considered as failed & fallbacks to the regular execution in main thread.
        Default is 9999'),
      '#default_value' => $this->config('partytown.settings')->get('fallback_timeout'),
      '#min' => 0,
      '#max' => 1000000,
      '#step' => 1,
    ];

    $form['debug'] = [
      '#title' => $this->t('Debug'),
      '#type' => 'fieldset',
      '#tree' => TRUE,
    ];
    $form['debug']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $this->config('partytown.settings')->get('debug.enabled'),
      '#description' => $this->t('Enable debug mode'),
    ];
    $form['debug']['logging'] = [
      '#title' => $this->t('Logging'),
      '#type' => 'fieldset',
    ];
    foreach (self::PARTYTOWN_LOG_TYPES as $log_type) {
      $form['debug']['logging'][$log_type] = [
        '#type' => 'checkbox',
        '#title' => $log_type,
        '#default_value' => $this->config('partytown.settings')->get("debug.logging.{$log_type}"),
        '#description' => $this->t("Enable {$log_type} logging"),
      ];
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * Submit callback to add another forward to the settings form.
   */
  public function addForward(array &$form, FormStateInterface $form_state) {
    $forward_count = $form_state->get('forward_count');
    $forward_plus_one = $forward_count + 1;
    $form_state->set('forward_count', $forward_plus_one);
    $form_state->setRebuild(TRUE);
  }

  /**
   * AJAX callback to rebuild the forwards wrapper in the settings form.
   */
  public function updateForwardAjaxCallback(array &$form, FormStateInterface $form_state) {
    return $form['forwards']['forwards_wrapper'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $settings = $this->config('partytown.settings');
    $settings
      ->set('enable', $form_state->getValue('enable'))
      ->set('lib', $form_state->getValue('lib'))
      ->set('debug.enabled', $form_state->getValue(['debug','enabled']))
      ->set('load_scripts_on_main_thread', $form_state->getValue('load_scripts_on_main_thread'))
      ->set('fallback_timeout', $form_state->getValue('fallback_timeout'));
    foreach (self::PARTYTOWN_LOG_TYPES as $log_type) {
      $settings->set("debug.logging.{$log_type}", $form_state->getValue(['debug','logging', $log_type]));
    }

    // Parse forwards.
    $all_values = $form_state->getValues();
    $forwards = [];
    foreach ($all_values as $key => $value) {
      if (strpos($key, 'function_') !== FALSE) {
        if (trim($value) == '') {
          continue;
        }
        $i = explode('_', $key)[1];
        $forwards[$i]['function'] = $value;
      }

      if (strpos($key, 'preserve_behavior_') !== FALSE) {
        $i = explode('_', $key)[2];
        if (!isset($forwards[$i])) {
          continue;
        }
        $forwards[$i]['preserve_behavior'] = $value;
      }
    }
    $settings->set('forward', $forwards);

    $settings->save();
    parent::submitForm($form, $form_state);
  }

}
