<?php

declare(strict_types=1);

namespace Drupal\partytown_drupal\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Partytown Drupal settings for this site.
 */
final class PartytownDrupalSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'partytown_drupal_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['partytown_drupal.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['enable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Partytown'),
      '#default_value' => $this->config('partytown_drupal.settings')->get('enable'),
      '#description' => $this->t('Enable the Partytown library and attach it to the page.'),
    ];

    $form['debug'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Debug Version'),
      '#default_value' => $this->config('partytown_drupal.settings')->get('debug'),
      '#description' => $this->t('When true, Partytown scripts are not inlined and not minified.
        <br>
        See the <a href="https://partytown.qwik.dev/debugging/" target="_blank">Debugging</a>
        docs on how to enable more logging.'),
    ];

    $form['lib'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Lib Path'),
      '#default_value' => $this->config('partytown_drupal.settings')->get('lib'),
      '#description' => $this->t('The path to the partytown lib directory,
        with leading and trailing slashes.
       Default is <code>/modules/contrib/partytown_drupal/lib/</code>'),
    ];

    $form['forwards'] = [
      '#type' => 'details',
      '#title' => 'Forwards',
      '#description' => $this->t('A list of function calls
        on the main thread to forward to the web worker.<br>
        See <a href="https://partytown.qwik.dev/forwarding-events" target="_blank">
        Forwarding Events and Triggers</a> for more info.'),
    ];

    $form['forwards']['actions'] = [
      '#type' => 'actions',
    ];

    $form['forwards']['actions']['add'] = [
      '#type' => 'submit',
      '#value' => $this->t('+ Add'),
      '#weight' => 100,
      '#submit' => ['::addForward'],
      '#ajax' => [
        'callback' => '::updateForwardAjaxCallback',
        'disable-refocus' => FALSE,
        'event' => 'click',
        'wrapper' => 'forwards-wrapper-ajax',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Adding forward...'),
        ],
      ],
    ];

    $form['forwards']['forwards_wrapper'] = [
      '#type' => 'fieldset',
      '#prefix' => '<div id="forwards-wrapper-ajax">',
      '#suffix' => '</div>',
    ];

    $forward_count = $form_state->get('forward_count');
    $forwards = $this->config('partytown_drupal.settings')->get('forward') ?: [];
    if (empty($forward_count)) {
      $forward_count = count($forwards) ?: 1;
      $form_state->set('forward_count', $forward_count);
    }

    // Render the current values, if any.
    for ($i = 0; $i < $forward_count; $i++) {
      $default_value = $form_state->getValue(['forwards', 'forwards_wrapper', 'function_' . $i])
        ?? ($forwards[$i]['function'] ?? '');
      $form['forwards']['forwards_wrapper']['function_' . $i] = [
        '#type' => 'textfield',
        '#title' => 'Function',
        '#description' => $this->t('The name of a function call to forward<br>
          ex. <code>dataLayer.push</code>'),
        '#default_value' => $default_value,
      ];

      $default_value = $form_state->getValue(['forwards', 'forwards_wrapper', 'preserve_behavior_' . $i])
        ?? ($forwards[$i]['preserve_behavior'] ?? '');
      $form['forwards']['forwards_wrapper']['preserve_behavior_' . $i] = [
        '#type' => 'checkbox',
        '#title' => 'Preserve Behavior',
        '#description' => $this->t('When preserveBehavior is set to true,
          the original function\'s behavior on the main thread is maintained,
          while also forwarding the calls to partytown.<br>
          See <a href="https://partytown.qwik.dev/forwarding-events/#preservebehavior" target="_blank">
          Forwarding Events</a> for more information.'),
        '#suffix' => '<hr>',
        '#default_value' => $default_value,
      ];
    }

    $form['load_scripts_on_main_thread'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Load Scripts On Main Thread'),
      '#description' => $this->t('An array of strings or regular expressions
        (RegExp) used to filter out which script are executed via Partytown
        and the main thread.<br>
        <br>
        An example is as follows:<br>
        <code>https://test.com/analytics.js</code><br>
        <code>inline-script-id</code><br>
        <code>/regex-matched-script\.js/</code><br>
        <br>
        Separate each rule with a newline.'),
      '#default_value' => $this->config('partytown_drupal.settings')->get('load_scripts_on_main_thread'),
    ];

    $form['resolve_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Resolve URL'),
      '#description' => $this->t('Javascript hook that is called to resolve
        URLs which can be used to modify URLs.<br>
        <br>
        The hook uses the API:
        <br>
        <code>resolveUrl(url: URL, location: URL, method: string)</code>.<br>
        See the <a href="https://partytown.qwik.dev/proxying-requests/" target="_blank">Proxying Requests</a>
        for more information.'),
      '#default_value' => $this->config('partytown_drupal.settings')->get('resolve_url'),
    ];

    $form['fallback_timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Fallback Timeout'),
      '#description' => $this->t('A timeout in ms until Partytown initialization is
        considered as failed & fallbacks to the regular execution in main thread.
        Default is 9999'),
      '#default_value' => $this->config('partytown_drupal.settings')->get('fallback_timeout'),
      '#min' => 0,
      '#max' => 1000000,
      '#step' => 1,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Submit callback to add another forward to the settings form.
   */
  public function addForward(array &$form, FormStateInterface $form_state) {
    $forward_count = $form_state->get('forward_count');
    $forward_plus_one = $forward_count + 1;
    $form_state->set('forward_count', $forward_plus_one);
    $form_state->setRebuild(TRUE);
  }

  /**
   * AJAX callback to rebuild the forwards wrapper in the settings form.
   */
  public function updateForwardAjaxCallback(array &$form, FormStateInterface $form_state) {
    return $form['forwards']['forwards_wrapper'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('partytown_drupal.settings');

    $config->set('enable', $form_state->getValue('enable'));
    $config->set('debug', $form_state->getValue('debug'));
    $config->set('lib', $form_state->getValue('lib'));
    $config->set('load_scripts_on_main_thread', $form_state->getValue('load_scripts_on_main_thread'));
    $config->set('resolve_url', $form_state->getValue('resolve_url'));
    $config->set('fallback_timeout', $form_state->getValue('fallback_timeout'));

    // Parse forwards.
    $all_values = $form_state->getValues();
    $forwards = [];
    foreach ($all_values as $key => $value) {
      if (strpos($key, 'function_') !== FALSE) {
        if (trim($value) == '') {
          continue;
        }
        $i = explode('_', $key)[1];
        $forwards[$i]['function'] = $value;
      }

      if (strpos($key, 'preserve_behavior_') !== FALSE) {
        $i = explode('_', $key)[2];
        if (!isset($forwards[$i])) {
          continue;
        }
        $forwards[$i]['preserve_behavior'] = $value;
      }
    }
    $config->set('forward', $forwards);

    $config->save();

    // Flush the site cache for this to take effect.
    drupal_flush_all_caches();
    parent::submitForm($form, $form_state);
  }

}
