<?php

namespace Drupal\password_reset_code\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\password_reset_code\Service\PasswordResetCodeManager;
use Drupal\user\Entity\User;
use Drupal\user\UserInterface;

class PasswordResetPasswordForm extends FormBase {

  public function __construct(protected PasswordResetCodeManager $manager) {}

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('password_reset_code.manager')
    );
  }

  public function getFormId() {
    return 'password_reset_code_password_form';
  }

  public function buildForm(array $form, FormStateInterface $form_state, $uuid = NULL, $code = NULL) {
    $row = $this->manager->loadByUuid((string) $uuid);
    // @todo check row exists.
    $now = \Drupal::time()->getRequestTime();

    if (!$row) {
      $this->messenger()->addError($this->t('This password reset request could not be found.'));
      return $form;
    }

    if (!hash_equals($row['code'], (string) $code)) {
      $this->messenger()->addError($this->t('The provided code is invalid.'));
      return $form;
    }

    if ((int) $row['validated'] !== 1) {
      $this->messenger()->addError($this->t('You must validate your code first.'));
      return $form;
    }

    if ((int) $row['expires'] < $now) {
      $this->manager->delete($row['uuid']);
      $this->messenger()->addError($this->t('Your password reset session has expired. Please start again.'));
      return $form;
    }

    $form['uuid'] = [
      '#type' => 'hidden',
      '#value' => $row['uuid'],
    ];

    $form['code'] = [
      '#type' => 'hidden',
      '#value' => (string) $code,
    ];

    $form['account'] = [
      '#type' => 'value',
      '#value' => (int) $row['uid'],
    ];

    $form['password'] = [
      '#type' => 'password_confirm',
      '#required' => TRUE,
      '#size' => 25,
      '#description' => $this->t('Choose a new password for your account.'),
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Set password and log in'),
    ];

    return $form;
  }

  public function validateForm(array &$form, FormStateInterface $form_state) {
    $uid = (int) $form_state->getValue('account');
    $account = User::load($uid);
    if (!$account) {
      $form_state->setErrorByName('password', $this->t('The account could not be loaded.'));
      return;
    }

    $passwords = $form_state->getValue('password');
    $pass = (string) ($passwords['pass1'] ?? '');

    // Check if password_policy module is installed and enabled.
    $moduleHandler = \Drupal::service('module_handler');
    $is_enabled = $moduleHandler->moduleExists('password_policy');
    if ($is_enabled) {
      if (\Drupal::service('password_policy.validation_manager')->validationShouldRun()) {
        /** @var \Drupal\password_policy\PasswordPolicyValidationReport $validationReport */
        $validationReport = \Drupal::service('password_policy.validator')->validatePassword(
          $form_state->getValue('password', ''),
          $account,
          $account->getRoles(),
        );

        if ($validationReport->isInvalid()) {
          $form_state->setErrorByName('password', t('The password does not satisfy the password policies.'));
          $form_state->setError($form, $validationReport->getErrors());
        }
      }
    }

    // Temporarily set the password to validate constraints (e.g., password_policy).
    $account->setPassword($pass);
    $violations = $account->validate();
    if (count($violations)) {
      foreach ($violations as $violation) {
        // Only show password-related messages.
        if (str_contains((string) $violation->getPropertyPath(), 'pass')) {
          $form_state->setErrorByName('password', $violation->getMessage());
        }
      }
    }
  }

  public function submitForm(array &$form, FormStateInterface $form_state) {
    $uuid = (string) $form_state->getValue('uuid');
    $uid = (int) $form_state->getValue('account');
    $account = User::load($uid);
    if (!$account instanceof UserInterface) {
      $this->messenger()->addError($this->t('The account could not be loaded.'));
      return;
    }

    $passwords = $form_state->getValue('password');
    $pass = (string) ($passwords['pass1'] ?? '');

    $account->setPassword($pass);
    $account->save();

    // Clean up the row to prevent reuse.
    $this->manager->delete($uuid);

    // Log the user in and redirect to their account page.
    user_login_finalize($account);
    $form_state->setRedirect('user.page');
  }
}
