<?php

namespace Drupal\password_reset_code\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\password_reset_code\Service\PasswordResetCodeManager;
use Drupal\Core\Url;

class PasswordResetCodeForm extends FormBase {

  public function __construct(
    protected PasswordResetCodeManager $manager,
  ) {}

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('password_reset_code.manager'),
    );
  }

  public function getFormId() {
    return 'password_reset_code_form';
  }

  public function buildForm(array $form, FormStateInterface $form_state, $uuid = NULL) {
    $config = $this->config('password_reset_code.settings');
    $max_tries = (int) $config->get('max_tries') ?: 5;
    $row = $this->manager->loadByUuid((string) $uuid);

    if (!$row) {
      $this->messenger->addWarning($this->t('Your password reset request could not be found. Please request a new one.'));
      return $this->redirectForm($form, Url::fromRoute('user.pass'));
    }

    $now = \Drupal::time()->getRequestTime();
    if ((int) $row['validated'] === 1) {
      $this->messenger->addWarning($config->get('msg_already_used') ?: $this->t('This password reset code has already been used.'));
      return $form;
    }
    if ((int) $row['expires'] < $now) {
      $this->manager->delete($row['uuid']);
      $this->messenger->addError($config->get('msg_expired') ?: $this->t('Your password reset code has expired. Please request a new one.'));
      return $form;
    }
    if ((int) $row['tries'] >= $max_tries) {
      $this->manager->delete($row['uuid']);
      $this->messenger->addError($config->get('msg_too_many_tries') ?: $this->t('Too many incorrect attempts. Please request a new password reset.'));
      return $form;
    }

    $form['instructions'] = [
      '#markup' => '<p>' . $this->t('Enter the 6-digit code we sent to your email.') . '</p>',
    ];

    $form['code'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Code'),
      '#required' => TRUE,
      '#size' => 6,
      '#maxlength' => 6,
      '#attributes' => [
        'pattern' => '[0-9]{6}',
        'inputmode' => 'numeric',
        'autocomplete' => 'one-time-code',
      ],
    ];

    $form['uuid'] = [
      '#type' => 'hidden',
      '#value' => $row['uuid'],
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Validate code'),
      '#attributes' => [
        'disabled' => 'disabled',
      ],
    ];

    $form['#attached']['library'][] = 'password_reset_code/code_input';

    return $form;
  }

  public function validateForm(array &$form, FormStateInterface $form_state) {
    $uuid = (string) $form_state->getValue('uuid');
    $code = preg_replace('/\D+/', '', (string) $form_state->getValue('code'));

    $config = $this->config('password_reset_code.settings');
    $max_tries = (int) $config->get('max_tries') ?: 5;

    $row = $this->manager->loadByUuid($uuid);
    if (!$row) {
      $form_state->setErrorByName('code', $this->t('This password reset request is no longer valid.'));
      return;
    }

    $now = \Drupal::time()->getRequestTime();
    if ((int) $row['expires'] < $now) {
      $this->manager->delete($row['uuid']);
      $form_state->setErrorByName('code', $this->t('Your password reset code has expired.'));
      return;
    }

    if ((int) $row['tries'] >= $max_tries) {
      $this->manager->delete($row['uuid']);
      $form_state->setErrorByName('code', $this->t('Too many incorrect attempts.'));
      return;
    }

    if (!$this->manager->checkCode($uuid, $code)) {
      $tries = $this->manager->incrementTries($uuid);
      $remaining = max(0, $max_tries - $tries);
      $form_state->setErrorByName('code', $this->t('The code you entered is not correct. @remaining attempts remaining.', ['@remaining' => $remaining]));
    }
    else {
      // Save normalized code back for submit.
      $form_state->setValue('code', $code);
    }
  }

  public function submitForm(array &$form, FormStateInterface $form_state) {
    $uuid = (string) $form_state->getValue('uuid');
    $code = (string) $form_state->getValue('code');

    // Mark as validated and extend expiry by 5 minutes (configurable).
    $extend = (int) $this->config('password_reset_code.settings')->get('extend_minutes') ?: 5;
    $this->manager->markValidatedAndExtend($uuid, $extend);

    $url = Url::fromRoute('password_reset_code.password_reset_password', ['uuid' => $uuid, 'code' => $code]);
    $form_state->setRedirectUrl($url);
  }

  protected function redirectForm(array $form, Url $url) {
    $form['redirect'] = [
      '#type' => 'markup',
      '#markup' => '',
      '#attached' => [
        'html_head' => [
          [['#tag' => 'meta', '#attributes' => ['http-equiv' => 'refresh', 'content' => '0;url=' . $url->toString()]], 'meta_redirect'],
        ],
      ],
    ];
    return $form;
  }
}
