<?php

namespace Drupal\password_reset_code\Service;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\user\UserInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Manager for password reset codes.
 */
class PasswordResetCodeManager {
  use StringTranslationTrait;

  public function __construct(
    protected Connection $database,
    protected TimeInterface $time,
    protected UuidInterface $uuidService,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerChannelInterface $logger,
    protected $stringTranslation,
    protected $routerBuilder,
    protected $routeProvider,
    protected ModuleHandlerInterface $moduleHandler,
  ) {}

  /**
   * Generate a 6-digit code where no digit appears more than twice.
   */
  public function generateCode(): string {
    while (TRUE) {
      $code = '';
      for ($i = 0; $i < 6; $i++) {
        $code .= random_int(0, 9);
      }
      $counts = count_chars($code, 1);
      $ok = TRUE;
      foreach ($counts as $char => $count) {
        if ($count > 2) {
          $ok = FALSE;
          break;
        }
      }
      if ($ok) {
        return $code;
      }
    }
  }

  /**
   * Returns password reset timeout from user.settings in seconds.
   */
  public function getResetTimeout(): int {
    return (int) $this->configFactory->get('user.settings')->get('password_reset_timeout') ?: 86400;
  }

  /**
   * Creates a record for a user and returns [uuid, code, expires].
   */
  public function createRecord(UserInterface $account): array {
    $uuid = $this->uuidService->generate();
    $code = $this->generateCode();
    $now = $this->time->getRequestTime();
    $expires = $now + $this->getResetTimeout();

    // Delete any existing records for this user to avoid clutter.
    $this->database->delete('user_pswd_code')->condition('uid', $account->id())->execute();

    $this->database->insert('user_pswd_code')->fields([
      'uid' => $account->id(),
      'created' => $now,
      'expires' => $expires,
      'uuid' => $uuid,
      'code' => $code,
      'tries' => 0,
      'validated' => 0,
    ])->execute();

    return [$uuid, $code, $expires];
  }

  /**
   * Load row by uuid.
   */
  public function loadByUuid(string $uuid): ?array {
    $row = $this->database->select('user_pswd_code', 'u')
      ->fields('u')
      ->condition('uuid', $uuid)
      ->execute()
      ->fetchAssoc();
    return $row ?: NULL;
  }

  /**
   * Verify code equality; does not change tries.
   */
  public function checkCode(string $uuid, string $code): bool {
    $row = $this->loadByUuid($uuid);
    return $row && hash_equals($row['code'], $code);
  }

  /**
   * Increment tries counter, returns updated value.
   */
  public function incrementTries(string $uuid): int {
    $this->database->update('user_pswd_code')
      ->expression('tries', 'tries + 1')
      ->condition('uuid', $uuid)
      ->execute();
    $row = $this->loadByUuid($uuid);
    return (int) ($row['tries'] ?? 0);
  }

  /**
   * Mark as validated and extend expiry by given minutes (default 5).
   */
  public function markValidatedAndExtend(string $uuid, int $minutes = 5): void {
    $new_expiry = $this->time->getRequestTime() + ($minutes * 60);
    $this->database->update('user_pswd_code')
      ->fields(['validated' => 1, 'expires' => $new_expiry])
      ->condition('uuid', $uuid)
      ->execute();
  }

  /**
   * Delete a record by uuid.
   */
  public function delete(string $uuid): void {
    $this->database->delete('user_pswd_code')->condition('uuid', $uuid)->execute();
  }

  /**
   * Build the URL to enter code.
   */
  public function buildCodeUrl(string $uuid): Url {
    return Url::fromRoute('password_reset_code.password_reset_code', ['uuid' => $uuid], ['absolute' => TRUE]);
  }

  /**
   * Build the URL to set password.
   */
  public function buildPasswordUrl(string $uuid, string $code): Url {
    return Url::fromRoute('password_reset_code.password_reset_password', ['uuid' => $uuid, 'code' => $code], ['absolute' => TRUE]);
  }
}
