<?php

namespace Drupal\patch_info\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;

/**
 * Config form to manage composer path and fetch API data.
 */
class ApiFetchConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'patch_info_details_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['patch_info.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('patch_info.settings');

    $form['ajax_message'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'ajax-message-wrapper'],
    ];

    $form['composer_location'] = [
      '#type' => 'radios',
      '#title' => $this->t('Composer.json location'),
      '#options' => [
        'webroot' => $this->t('Webroot') . ' (' . DRUPAL_ROOT . '/composer.json)',
        'parent' => $this->t('One level above webroot') . ' (' . dirname(DRUPAL_ROOT) . '/composer.json)',
        'custom' => $this->t('Custom path'),
      ],
      '#default_value' => $config->get('composer_location') ?: 'webroot',
    ];

    $form['composer_custom_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom composer path'),
      '#default_value' => $config->get('composer_custom_path') ?: '',
      '#description' => $this->t('Enter full directory path containing composer.json'),
      '#size' => 60,
      '#states' => [
        'visible' => [
          ':input[name="composer_location"]' => ['value' => 'custom'],
        ],
      ],
    ];

    $form['check_composer'] = [
      '#type' => 'button',
      '#value' => $this->t('Test Composer Path'),
      '#ajax' => [
        'callback' => '::checkComposerPathAjax',
        'wrapper' => 'composer-check-result',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Checking composer.json...'),
        ],
      ],
    ];

    $form['composer_result'] = [
      '#type' => 'markup',
      '#attributes' => ['class' => ['drupal-success-field']],
      '#markup' => '<div id="composer-check-result"></div>',
    ];

    $form['api_section'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('API Fetch'),
    ];

    $form['api_section']['fetch_button'] = [
      '#type' => 'button',
      '#value' => $this->t('Fetch API Records'),
      '#ajax' => [
        'callback' => '::fetchRecordsAjax',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Fetching API data...'),
        ],
      ],
    ];

    $form['api_section']['results'] = [
      '#type' => 'markup',
      '#markup' => '<div id="api-results"></div>',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * AJAX callback to test composer.json file.
   */
  public function checkComposerPathAjax(array &$form, FormStateInterface $form_state) {
    $composer_location = $form_state->getValue('composer_location');

    $full_path = $this->getComposerPath($composer_location, $form_state);

    if (file_exists($full_path)) {
      $html = '<div class="messages messages--status">composer.json found at ' . htmlspecialchars($full_path) . '</div>';
    }
    else {
      $html = '<div class="messages messages--error">composer.json not found at ' . htmlspecialchars($full_path) . '</div>';
    }

    $response = new AjaxResponse();
    $response->addCommand(
       new HtmlCommand('#ajax-message-wrapper', $html)
    );

    return $response;
  }

  /**
   * AJAX callback to fetch API data.
   */
  public function fetchRecordsAjax(array &$form, FormStateInterface $form_state) {
    $nids = [];
    $response = new AjaxResponse();
    $composer_path = $this->getComposerPath($this->config('patch_info.settings')->get('composer_location'), NULL);
    if (!file_exists($composer_path)) {
      $html = '<div class="messages messages--error">composer.json not found at ' . htmlspecialchars($composer_path) . '</div>';
      $response->addCommand(
        new HtmlCommand('#ajax-message-wrapper', $html)
      );
      return $response;
    }

    // Read composer.json file and extract NIDs from patch URLs.
    $contents = file_get_contents($composer_path);
    $composer = json_decode($contents, TRUE);
    if (isset($composer['extra']['patches']) && count($composer['extra']['patches'])) {
      foreach ($composer['extra']['patches'] as $patches) {
        foreach ($patches as $patch_url) {
          $filename = basename($patch_url);
          if (preg_match('/\b\d{7,}\b/', $filename, $matches)) {
            if ($matches[0]) {
              $nids[] = $matches[0];
            }
          }
        }
      }
    }

    if (count($nids)) {
      $nids = array_unique($nids);
      /** @var \Drupal\patch_info\Api\PatchInfoService $fetcher */
      $fetcher = \Drupal::service('patch_info.patch_info_service');
      $fetcher->batchProcessIssues($nids);
    }

    $html = '<div class="messages messages--status">Patch data fetched successfuly.</div>';
    $response->addCommand(
      new HtmlCommand('#ajax-message-wrapper', $html)
    );
    return $response;

  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('patch_info.settings')
      ->set('composer_location', $form_state->getValue('composer_location'))
      ->set('composer_custom_path', $form_state->getValue('composer_custom_path'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get full path to composer.json based on selected location.
   */
  public function getComposerPath($location_option, ?FormStateInterface $form_state) {
    switch ($location_option) {
      case 'webroot':
        return DRUPAL_ROOT . '/composer.json';

      case 'parent':
        return dirname(DRUPAL_ROOT) . '/composer.json';

      case 'custom':
        return ($form_state) ? $form_state->getValue('composer_custom_path') : $this->config('patch_info.settings')->get('composer_custom_path');

      default:
        return DRUPAL_ROOT . '/composer.json';
    }
  }

}
