<?php

namespace Drupal\patch_info\Api;

use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Component\Datetime\TimeInterface;
use GuzzleHttp\ClientInterface;

/**
 * Fetches entities from an external API and stores them in DB.
 */
class PatchInfoService {

  protected ClientInterface $httpClient;
  protected Connection $database;
  protected TimeInterface $time;
  protected $logger;
  protected array $issue_data = [];

  /**
   * Constructor.
   */
  public function __construct(ClientInterface $http_client, Connection $database, TimeInterface $time, LoggerChannelFactoryInterface $logger_factory) {
    $this->httpClient = $http_client;
    $this->database = $database;
    $this->time = $time;
    $this->logger = $logger_factory->get('patch_info');
  }

  /**
   * Fetch data from API and insert into DB.
   */
  public function fetchAndInsert(string $api_url): void {
    try {
      $response = $this->httpClient->get($api_url, ['timeout' => 500]);
      if ($response->getStatusCode() === 200) {
        $data = json_decode($response->getBody(), TRUE);
        if (!empty($data) && !empty($data['list'])) {
          foreach ($data['list'] as $item) {
            if (preg_match('#project/([^/]+)/issues#', $item['url'], $matches)) {
              $item['project_name'] = $matches[1] != 'drupal' ? $matches[1] : 'core';
            }

            // Optional: avoid duplicates.
            $exists = $this->database->select('patch_info_entities', 'm')
              ->fields('m', ['id'])
              ->condition('entity_name', $item['project_name'])
              ->condition('url', $item['url'])
              ->execute()
              ->fetchField();

            if (!$exists) {
              $this->database->insert('patch_info_entities')
                ->fields([
                  'entity_name' => $item['project_name'] ?? '',
                  'description' => strip_tags($item['title'] ?? ''),
                  'issue_id' => $item['nid'] ?? 0,
                  'url' => $item['url'] ?? '',
                  'type' => $item['type'] ?? '',
                  'patch_url' => $file_name = $this->issue_data[$item['nid']] ?? '',
                  'patch_filename' => basename($file_name),
                  'status' => $item['field_issue_status'] ?? '',
                  'created' => $item['created'] ?? time(),
                ])
                ->execute();
            }
          }
        }
      }
    }
    catch (\Exception $e) {
      $this->logger->error('API fetch failed: @message', ['@message' => $e->getMessage()]);
    }
  }

  /**
   * Clear all data from the patch_info_entities table.
   */
  public function clearData(): void {
    $this->database->truncate('patch_info_entities')->execute();
  }

  /**
   * Build the API URL with given issue IDs.
   */
  public function buildApiUrl(array $issue_ids): string {
    $api_url = 'https://www.drupal.org/api-d7/node.json?type=project_issue';
    foreach ($issue_ids as $nid) {
      $api_url .= '&nid[]=' . intval($nid);
    }
    return $api_url;
  }

  /**
   * Process issue IDs in batches.
   */
  public function batchProcessIssues(array $issue_data, $batch_size = 10): void {
    $this->clearData();
    $this->issue_data = $issue_data;
    $issue_ids = array_keys($issue_data);
    $batches = array_chunk($issue_ids, $batch_size);
    foreach ($batches as $batch) {
      $api_url = $this->buildApiUrl($batch);
      $this->fetchAndInsert($api_url);
    }
  }

  /**
   * Process for custom patches.
   */
  public function customPatchProcess(array $custom_patch_data): void {
    // Handle custom patch data as needed.
    if (count($custom_patch_data)) {
      foreach ($custom_patch_data as $title => $url) {
        if (preg_match('/\bhttps?:\/\/[^\s]+$/i', $url)) {
          $entity_name = 'unknown';
          $type = 'unknown_status_patch';
          $status = '-2';
        }
        else {
          $entity_name = 'custom';
          $type = 'custom_patch';
          $status = '-1';
        }

        // Optional: avoid duplicates.
        $exists = $this->database->select('patch_info_entities', 'm')
          ->fields('m', ['id'])
          ->condition('entity_name', $entity_name)
          ->condition('url', $url)
          ->execute()
          ->fetchField();

        if (!$exists) {
          $this->database->insert('patch_info_entities')
            ->fields([
              'entity_name' => $entity_name ?? '',
              'description' => strip_tags($title ?? ''),
              'issue_id' => 0,
              'url' => $url ?? '',
              'type' => $type ?? '',
              'patch_url' => $url ?? '',
              'patch_filename' => basename($url),
              'status' => $status ?? '',
              'created' => time(),
            ])
            ->execute();
        }
      }
    }
  }

}
