<?php

declare(strict_types=1);

namespace Drupal\path_alias_views_cshs_filter\Plugin\views\filter;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormStateInterface;
use Drupal\cshs\Component\CshsOption;
use Drupal\views\Attribute\ViewsFilter;
use Drupal\views\Plugin\views\filter\StringFilter;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Views filter for paths.
 *
 * Uses the CSHS hierarchical select widget to allow drilling down subpaths.
 */
#[ViewsFilter('path_alias_alias')]
class PathAliasAlias extends StringFilter {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('database'),
      $container->get('cache.default'),
    );
  }

  /**
   * Creates a PathAliasAlias instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    Connection $connection,
    protected CacheBackendInterface $cache,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $connection);
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();

    $options['path_depth'] = ['default' => 1];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function hasExtraOptions() {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function buildExtraOptionsForm(&$form, FormStateInterface $form_state) {
    $form['path_depth'] = [
      '#type' => 'number',
      '#min' => 1,
      '#step' => 1,
      '#title' => $this->t('Path depth'),
      '#description' => $this->t('The number of path components to show in the filter.'),
      '#default_value' => $this->options['path_depth'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function valueForm(&$form, FormStateInterface $form_state) {
    // Cache the aliases for the filter, as this is a messy query with
    // potentially a big result set.
    $cid = 'path_alias_views.path_alias_alias';
    if ($cached = $this->cache->get($cid)) {
      $aliases = $cached->data;
    }
    else {
      // Query for all aliases up to the path depth.
      $query = \Drupal::database()
        ->select('path_alias')
        ->fields('path_alias', ['alias'])
        ->condition('path_alias.status', 1)
        ->where("LENGTH(path_alias.alias) - LENGTH(REPLACE(path_alias.alias, '/', '')) <= :depth", [
          ':depth' => $this->options['path_depth'],
        ])
        ->orderBy('path_alias.alias');
      $result = $query->execute();
      $aliases = $result->fetchAllKeyed(0, 0);

      // Use the 'rendered' cache tag as that is cleared when entities are
      // saved.
      // WARNING: Editing aliases in the path alias UI won't clear this because
      // of a core bug.
      // @see https://www.drupal.org/project/drupal/issues/2480077.
      $this->cache->set($cid, $aliases, Cache::PERMANENT, ['rendered']);
    }

    $options = [];
    foreach ($aliases as $alias) {
      $pieces = explode('/', $alias);

      // Top-level path.
      if (count($pieces) == 2) {
        // Trim the initial '/' as it's shown with CSS before the SELECT
        // element.
        $alias_label = ltrim($alias, '/');

        $options[$alias] =  new CshsOption($alias_label);

        continue;
      }
      else {
        // Get the parent of this alias: for example, if the alias is /a/b/c, we
        // want to find /a/b. Remove the last path component (would be nice if
        // we could limit explode() from the right, but we can't).
        $parent_path = preg_replace('@/[^/]+$@', '', $alias);


        // Skip this path if the derived parent path is not an alias itself.
        if (!isset($aliases[$parent_path])) {
          continue;
        }

        // Truncate the parent from the alias for the label.
        $alias_label = substr($alias, strlen($parent_path));

        // Trim the initial '/' as it's shown with CSS before the SELECT
        // element.
        $alias_label = ltrim($alias_label, '/');

        $options[$alias] =  new CshsOption($alias_label, $parent_path);
      }
    }

    $form['value'] = [
      '#type' => 'cshs',
      '#title' => $this->t('Value'),
      '#options' => $options,
      '#default_value' => $this->value,
      '#none_label' => $this->t('- Select subpath -'),
      '#attributes' => [
        'class' => [
          'path-alias-views-alias-filter-cshs',
        ],
      ],
      '#attached' => [
        'library' => [
          'path_alias_views_cshs_filter/path_alias_filter',
        ],
      ],
    ];

    if ($form_state->get('exposed')) {
      $identifier = $this->options['expose']['identifier'];
      $user_input = $form_state->getUserInput();
      if (!isset($user_input[$identifier])) {
        $user_input[$identifier] = $this->value;
        $form_state->setUserInput($user_input);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function operators() {
    $operators = [
      '=' => [
        'title' => $this->t('Is equal to'),
        'short' => $this->t('='),
        'method' => 'opEqual',
        'values' => 1,
      ],
      // @todo Make this the default operator.
      'starts' => [
        'title' => $this->t('Starts with'),
        'short' => $this->t('begins'),
        'method' => 'opStartsWith',
        'values' => 1,
      ],
    ];

    return $operators;
  }

}
