<?php

namespace Drupal\pci_sri\Drush\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\pci_sri\PciSriExtensionDiscovery;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Drush command for generating and updating the SRI inventory.
 */
class PciSriCommands extends DrushCommands {

  /**
   * Constructs a PciSriCommands object.
   */
  public function __construct(
    private readonly PciSriExtensionDiscovery $pciSriExtensionDiscovery,
    private readonly Yaml $yaml,
    private readonly ModuleHandlerInterface $moduleHandler,
    private readonly ThemeHandlerInterface $themeHandler,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('pci_sri.extension_discovery'),
      $container->get('pci_sri.yaml'),
      $container->get('module_handler'),
      $container->get('theme_handler'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Generate the SRI configuration.
   */
  #[CLI\Command(name: 'sri:generate', aliases: ['sri-gen'])]
  #[CLI\Usage(name: 'sri:generate', description: 'Usage description')]
  public function generate() {
    // Get a list of enabled modules.
    $modules = $this->pciSriExtensionDiscovery->scan('module');
    $enabled_modules = $this->getEnabledModules($modules);

    // Get a list of enabled themes.
    $themes = $this->pciSriExtensionDiscovery->scan('theme');
    $enabled_themes = $this->getEnabledThemes($themes);

    // Gather all the libraries from the modules and themes.
    $module_libraries = $this->getLibraries($enabled_modules);
    $theme_libraries = $this->getLibraries($enabled_themes);

    // Gather all the Javascript libraries from the modules abd themes.
    $module_js_libraries = $this->getJavascriptLibraries($module_libraries, 'module');
    $theme_js_libraries = $this->getJavascriptLibraries($theme_libraries, 'theme');
    $js_libraries = array_merge($module_js_libraries, $theme_js_libraries);

    $sri_libraries = [];

    // Determine if the SRI is enabled for the Javascript.
    foreach ($js_libraries as $library_name => $library_paths) {
      foreach ($library_paths as $library_path) {
        $module = $library_path['name'];
        $sri_enabled = $this->isSriEnabled($library_path['path'], $library_path['preprocess']);

        $sri_libraries[] = [
          'sri_enabled' => $sri_enabled ? 'Y' : 'N',
          'module' => $module,
          'library_name' => $library_name,
          'path' => $library_path['path'],
        ];
      }
    }

    // Generate the SRI configuration.
    $sri_configuration_added = FALSE;
    $sri_configuration_updated = FALSE;

    foreach ($sri_libraries as $sri_library) {
      $js_filename = basename($sri_library['path']);

      // Generate a machine for the SRI configuration ID.
      $module = $sri_library['module'];
      $library_name = $sri_library['library_name'];
      $label = $js_filename . ' (' . $module . '/' . $library_name . ')';
      $machine_name = preg_replace('/[^a-z0-9_]+/', '_', strtolower($label));

      // Limit the machine name to 64 characters.
      $machine_name = substr($machine_name, 0, 64);

      // Generate an SHA-512 hash for the Javascript file.
      if (file_exists($sri_library['path'])) {
        $data = file_get_contents($sri_library['path']);
        $hash = 'sha512-' . base64_encode(hash('sha512', $data, TRUE));
      }
      else {
        // We don't generate a hash for cloud-based Javascript files.
        $hash = '';
      }

      // Lookup the SRI configuration by the Javascript file path.
      $sri = $this->entityTypeManager->getStorage('sri')->loadByProperties(['js_file_path' => $sri_library['path']]);

      // If the SRI configuration does not exist, create it.
      if (empty($sri)) {
        $sri = $this->entityTypeManager->getStorage('sri')->create([
          'id' => $machine_name,
          'label' => $js_filename,
          'js_file_path' => $sri_library['path'],
          'sri_hash' => $hash,
          'cross_origin' => 'anonymous',
          'status' => $sri_library['sri_enabled'] === 'Y',
        ]);

        $sri_configuration_added = TRUE;
      }
      // Otherwise, update the existing SRI configuration.
      else {
        $sri = reset($sri);
        if ($sri->get('sri_hash') !== $hash) {
          $sri_configuration_updated = TRUE;
        }
        $sri->set('sri_hash', $hash);
        $sri->set('status', $sri_library['sri_enabled'] === 'Y');
      }

      $sri->save();
    }

    // Output the result of the SRI configuration generation.
    if ($sri_configuration_added) {
      $this->logger()->success(dt('SRI configuration has been added.'));
    }
    if ($sri_configuration_updated) {
      $this->logger()->success(dt('SRI configuration has been updated.'));
    }
    if (!$sri_configuration_added && !$sri_configuration_updated) {
      $this->logger()->success(dt('No SRI configuration changes were made.'));
    }
  }

  /**
   * Get a list of enabled modules.
   *
   * @param array $modules
   *   An array of Extension objects.
   *
   * @return array
   *   An array of enabled Extension objects.
   */
  protected function getEnabledModules(array $modules) {
    $enabled_modules = [];
    foreach ($modules as $module) {
      if ($this->moduleHandler->moduleExists($module->getName())) {
        $enabled_modules[] = $module;
      }
    }
    return $enabled_modules;
  }

  /**
   * Get a list of enabled themes.
   *
   * @param array $themes
   *   An array of Extension objects.
   *
   * @return array
   *   An array of enabled Extension objects.
   */
  protected function getEnabledThemes(array $themes) {
    $enabled_themes = [];
    foreach ($themes as $theme) {
      if ($this->themeHandler->themeExists($theme->getName())) {
        $enabled_themes[] = $theme;
      }
    }
    return $enabled_themes;
  }

  /**
   * Gather all the libraries from the modules or themes.
   *
   * @param array $extensions
   *   An array of Extension objects.
   *
   * @return array
   *   An array of libraries.
   */
  protected function getLibraries(array $extensions) {
    $libraries = [];
    foreach ($extensions as $extension) {
      $libraries_yml = $extension->getPath() . '/' . $extension->getName() . '.libraries.yml';
      if (file_exists($libraries_yml)) {
        $parsed_libraries = $this->yaml->parseFile($libraries_yml);
        foreach ($parsed_libraries as $lib_name => $lib_info) {
          $libraries[$extension->getName()][$lib_name] = $lib_info;
        }
      }
    }

    return $libraries;
  }

  /**
   * Gather the Javascript libraries from the modules or themes.
   *
   * @param array $extensions
   *   An array of Extension objects.
   * @param string $extension_type
   *   The type of extension.
   *
   * @return array
   *   An array of Javascript libraries.
   */
  protected function getJavascriptLibraries(array $extensions, string $extension_type) {
    $js_libraries = [];

    foreach ($extensions as $extension_name => $library) {
      if ($extension_type === 'module') {
        $extension_path = $this->moduleHandler->getModule($extension_name)->getPath();
      }
      elseif ($extension_type === 'theme') {
        $extension_path = $this->themeHandler->getTheme($extension_name)->getPath();
      }

      foreach ($library as $library_name => $library_data) {
        if (isset($library_data['js']) && is_array($library_data['js'])) {
          foreach ($library_data['js'] as $js_path => $js_attr) {
            $path = $this->getFullJavascriptPath($extension_path, $js_path);

            if (isset($js_attr['preprocess'])) {
              $preprocess = $js_attr['preprocess'];
            }
            else {
              $preprocess = 0;
            }

            $js_libraries[$library_name][] = [
              'name' => $extension_name,
              'path' => $path,
              'preprocess' => $preprocess,
            ];
          }
        }
      }
    }

    return $js_libraries;
  }

  /**
   * Return the full path of the Javascript library.
   *
   * @param string $module_or_theme_path
   *   The path of the module or theme.
   * @param string $js_path
   *   The path of the Javascript file.
   *
   * @return string
   *   The full path of the Javascript file.
   */
  protected function getFullJavascriptPath(string $module_or_theme_path, $js_path) {
    if (str_starts_with($js_path, 'https:')) {
      $path = $js_path;
    }
    elseif (str_starts_with($js_path, '//')) {
      $path = $js_path;
    }
    else {
      // Ensure the module path and Javascript path are separated by backslash.
      if (str_starts_with($js_path, '/')) {
        $path = $module_or_theme_path . $js_path;
      }
      else {
        $path = $module_or_theme_path . '/' . $js_path;
      }
    }

    return $path;
  }

  /**
   * Check if SRI is to be enabled for the Javascript file.
   *
   * Enabled  - Javascript is not preprocessed.
   * Disabled - Javascript is preprocessed, or cloud-based.
   *
   * @param string $js_path
   *   The path of the Javascript file.
   * @param bool $preprocess
   *   Whether the Javascript file is preprocessed.
   *
   * @return bool
   *   TRUE if SRI is enabled, FALSE otherwise.
   */
  protected function isSriEnabled(string $js_path, bool $preprocess) {
    $sri_enabled = TRUE;

    if (str_starts_with($js_path, 'https:') || str_starts_with($js_path, '//')) {
      $sri_enabled = FALSE;
    }
    elseif ($preprocess) {
      $sri_enabled = FALSE;
    }

    return $sri_enabled;
  }

}
