<?php

namespace Drupal\pcp\Plugin\Block;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\pcp\PcpServiceInterface;
use Drupal\user\Entity\User;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'PCP' block.
 *
 * @Block(
 *   id = "pcp_block",
 *   admin_label = @Translation("Profile Complete Percentage"),
 *   category = @Translation("User")
 * )
 */
class PCPBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The current user account proxy.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $accountProxy;

  /**
   * The PCP Service.
   *
   * @var \Drupal\pcp\PcpServiceInterface
   */
  protected $pcpService;

  /**
   * PCPBlock constructor.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Session\AccountInterface $accountProxy
   *   The current user account proxy.
   * @param \Drupal\pcp\PcpServiceInterface $pcpService
   *   The PCP Service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, AccountInterface $accountProxy, PcpServiceInterface $pcpService) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->accountProxy = $accountProxy;
    $this->pcpService = $pcpService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
      $container->get('pcp.pcp_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function blockAccess(AccountInterface $account) {
    return $account->isAuthenticated() ? AccessResult::allowed() : AccessResult::forbidden();
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $user_id = $this->accountProxy->id();
    /** @var \Drupal\user\UserInterface $user */
    $user = User::load($user_id);

    $pcp_data = $this->pcpService->getCompletePercentageData($user);

    if (($pcp_data['hide_pcp_block'] && $pcp_data['incomplete'] == 0) || $pcp_data['total'] == 0) {
      return [];
    }

    $pcp_markup = [
      '#theme' => 'pcp_template',
      '#uid' => $pcp_data['uid'],
      '#total' => $pcp_data['total'],
      '#open_link' => $pcp_data['open_link'],
      '#completed' => $pcp_data['completed'],
      '#incomplete' => $pcp_data['incomplete'],
      '#next_percent' => $pcp_data['next_percent'],
      '#nextfield_name' => $pcp_data['nextfield_name'],
      '#nextfield_title' => $pcp_data['nextfield_title'],
      '#current_percent' => $pcp_data['current_percent'],
      '#attached' => [
        'library' => ['pcp/pcp.block'],
      ],
    ];

    return $pcp_markup;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    return 0;
  }

}
