<?php

namespace Drupal\pdf_forge\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Url;
use Drupal\pdf_forge\Service\PdfForgeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;
use Drupal\Core\Extension\ExtensionPathResolver;

/**
 * Controller for PDF Forge demo pages.
 *
 * Provides routes for:
 * - A demo hub page listing available demos.
 * - A basic PDF rendering demo.
 * - An advanced PDF rendering demo with header/footer, margins, and pagination.
 *
 * @package Drupal\pdf_forge\Controller
 */
final class PdfForgeDemoController extends ControllerBase {

  /**
   * PDF Forge service.
   *
   * @var \Drupal\pdf_forge\Service\PdfForgeInterface
   */
  private readonly PdfForgeInterface $forge;

  /**
   * Date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  private readonly DateFormatterInterface $dateFormatter;

  /**
   * The extension path resolver service.
   *
   * Used to determine the absolute filesystem path to the module's
   * directory so that bundled assets (e.g., CSS, images, fonts) can
   * be reliably located and referenced in generated PDFs.
   *
   * @var \Drupal\Core\Extension\ExtensionPathResolver
   */
  private readonly ExtensionPathResolver $extensionPathResolver;

  /**
   * Constructs a new PdfForgeDemoController.
   *
   * @param \Drupal\pdf_forge\Service\PdfForgeInterface $forge
   *   The PDF Forge service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   */
  public function __construct(
    PdfForgeInterface $forge,
    DateFormatterInterface $dateFormatter,
    ExtensionPathResolver $extensionPathResolver,
  ) {
    $this->forge = $forge;
    $this->dateFormatter = $dateFormatter;
    $this->extensionPathResolver = $extensionPathResolver;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('pdf_forge.forge'),
      $container->get('date.formatter'),
      $container->get('extension.path.resolver'),
    );
  }

  /**
   * Builds the demo index page listing available demos.
   *
   * @return array
   *   A render array representing the demo index table.
   */
  public function index(): array {
    $items = [
      [
        'title' => $this->t('Basic demo'),
        'desc'  => $this->t('Minimal render array → PDF. Good for sanity checks.'),
        'route' => Url::fromRoute('pdf_forge.demo_basic'),
      ],
      [
        'title' => $this->t('Advanced demo'),
        'desc'  => $this->t('Adds Word-style header/footer, page margins, and an SVG logo.'),
        'route' => Url::fromRoute('pdf_forge.demo_advanced'),
      ],
    ];

    $rows = [];
    foreach ($items as $it) {
      $rows[] = [
        ['data' => [
          '#type' => 'link',
          '#title' => $it['title'],
          '#url' => $it['route'],
          '#options' => ['attributes' => []],
        ]],
        ['data' => [
          '#plain_text' => (string) $it['desc'],
        ]],
      ];
    }

    $build['intro'] = [
      '#markup' => '<p>' . $this->t('Pick a demo:') . '</p>',
    ];

    $build['table'] = [
      '#type'   => 'table',
      '#header' => [$this->t('Demo'), $this->t('What it shows')],
      '#rows'   => $rows,
      '#empty'  => $this->t('No demos found.'),
    ];

    return $build;
  }

  /**
   * Demonstrates a basic PDF rendering example.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The generated PDF as a file download response.
   */
  public function demoBasic(): Response {
    $build = [
      '#theme' => 'pdf_forge_print',
      '#title' => 'Hello PDF',
      '#css' => '',
      '#content' => [
        '#type' => 'inline_template',
        '#template' => '<p>This PDF was generated on {{ date }} by PDF Forge.</p><p><strong>It uses Dompdf under the hood.</strong></p>',
        '#context' => [
          'date' => $this->dateFormatter->format(\Drupal::time()->getRequestTime(), 'long'),
        ],
      ],
    ];

    return $this->forge->render($build, [
      'title' => 'demo-basic',
      'show_pagination' => true,
      'pagination_x' => 540,
      'pagination_y' => 820,
    ]);
  }

  /**
   * Demonstrates an advanced PDF rendering example.
   *
   * Includes a header and footer, custom page margins, and pagination.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The generated PDF as a file download response.
   */
  public function demoAdvanced(): Response {

    // Get the absolute path to the logo in the module's css/ folder.
    $logo_path = $this->extensionPathResolver
      ->getPath('module', 'pdf_forge') . '/css/test-logo.svg';

    // Convert local path to file:// for Dompdf. If its in a public path, do public://test-logo.svg
    $logo_src = 'file://' . DRUPAL_ROOT . '/' . $logo_path;

    $header = '<table width="100%" style="font-size:12px;">
      <tr>
        <td><img src=' . "$logo_src" . ' width="120" /></td>
        <td style="text-align:right;">
          <strong>Order #12345</strong><br/>' . $this->dateFormatter->format(\Drupal::time()->getRequestTime(), 'medium') . '
        </td>
      </tr>
    </table>';

    $footer = '<div style="font-size:10px; text-align:center;">
      Confidential — &copy; ' . date('Y') . ' My Company
    </div>';

    $build = [
      '#theme' => 'pdf_forge_print',
      '#title' => 'Invoice 12345',
      '#content' => [
        '#markup' => '<p>Total: $42</p>',
      ],
      '#css' => 'p{font-size:14px;}',
    ];

    return $this->forge->render($build, [
      'title' => 'demo-advanced',
      'page_size' => 'A4',
      'orientation' => 'portrait',
      'header_html' => $header,
      'header_height_mm' => 36,
      'footer_html' => $footer,
      'footer_height_mm' => 18,
      'show_pagination' => true,
      'pagination_x' => 540,
      'pagination_y' => 820,
    ]);
  }

}
