<?php

namespace Drupal\pdf_forge\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Defines the configuration form for PDF Forge.
 *
 * Allows site administrators to configure:
 * - Default font.
 * - Default page size and orientation.
 * - Remote asset permissions and allowlist.
 *
 * @package Drupal\pdf_forge\Form
 */
final class PdfForgeSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['pdf_forge.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'pdf_forge_settings_form';
  }

  /**
   * Builds the PDF Forge settings form.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The render array for the configuration form.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('pdf_forge.settings');

    $form['default_font'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default font'),
      '#default_value' => (string) ($config->get('default_font') ?? 'Open Sans'),
      '#description' => $this->t('Font must be available to Dompdf.'),
      '#required' => TRUE,
    ];

    $form['page_size'] = [
      '#type' => 'select',
      '#title' => $this->t('Default page size'),
      '#options' => [
        'A4' => 'A4',
        'Letter' => 'Letter',
        'Legal' => 'Legal',
        'A3' => 'A3',
      ],
      '#default_value' => (string) ($config->get('page_size') ?? 'A4'),
    ];

    $form['orientation'] = [
      '#type' => 'select',
      '#title' => $this->t('Default orientation'),
      '#options' => [
        'portrait' => $this->t('Portrait'),
        'landscape' => $this->t('Landscape'),
      ],
      '#default_value' => (string) ($config->get('orientation') ?? 'portrait'),
    ];

    $form['allow_remote_assets'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow remote assets (CSS/images)'),
      '#default_value' => (bool) $config->get('allow_remote_assets'),
      '#description' => $this->t('If enabled, Dompdf may request CSS/images from approved hosts below. This has security implications; keep the allowlist tight.'),
    ];

    $form['remote_allowlist'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Remote host allowlist'),
      '#default_value' => implode("\n", (array) $config->get('remote_allowlist') ?: []),
      '#description' => $this->t('One host per line, e.g. example.com'),
      '#states' => [
        'visible' => [
          ':input[name="allow_remote_assets"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * Saves the configuration values for PDF Forge.
   *
   * @param array $form
   *   The complete form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $hosts_raw = (string) ($form_state->getValue('remote_allowlist') ?? '');
    $hosts = preg_split('/\r?\n/', $hosts_raw) ?: [];
    $hosts = array_values(array_filter(array_map('trim', $hosts)));

    $this->configFactory->getEditable('pdf_forge.settings')
      ->set('default_font', (string) $form_state->getValue('default_font'))
      ->set('page_size', (string) $form_state->getValue('page_size'))
      ->set('orientation', (string) $form_state->getValue('orientation'))
      ->set('allow_remote_assets', (bool) $form_state->getValue('allow_remote_assets'))
      ->set('remote_allowlist', $hosts)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
