<?php

/**
 * @file
 * Contains the DomPdfForge service class.
 */

namespace Drupal\pdf_forge\Service;

use Dompdf\Adapter\CPDF;
use Dompdf\Dompdf;
use Dompdf\Options;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ExtensionPathResolver;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Render\Renderer;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\ClientInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;

/**
 * Service to generate PDFs from HTML using Dompdf.
 *
 * Handles:
 * - Rendering Drupal render arrays into HTML and then into PDF.
 * - Applying inline CSS, module CSS, and optional remote/local CSS.
 * - Header and footer injection with margin adjustments.
 * - Remote asset security via an allowlist.
 * - Pagination and PDF metadata configuration.
 */
final class DomPdfForge implements PdfForgeInterface {

  use StringTranslationTrait;

  /**
   * Public directory to cache Dompdf fonts.
   *
   * @var string
   */
  public const FONT_DIR = 'public://dompdf_fonts';

  /**
   * Constructs a DomPdfForge service.
   *
   * @param \Drupal\Core\Render\Renderer $renderer
   *   The Drupal renderer service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\Core\Extension\ExtensionPathResolver $extensionPathResolver
   *   Resolves module and theme paths.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client for fetching remote CSS files.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory service.
   */
  public function __construct(
    private readonly Renderer $renderer,
    private readonly RequestStack $requestStack,
    private readonly ModuleHandlerInterface $moduleHandler,
    private readonly ExtensionPathResolver $extensionPathResolver,
    private readonly FileSystemInterface $fileSystem,
    private readonly ClientInterface $httpClient,
    private readonly ConfigFactoryInterface $configFactory
  ) {}

  /**
   * {@inheritdoc}
   *
   * Generates a PDF response from a render array.
   *
   * @param array $build
   *   The Drupal render array to render as HTML and convert to PDF.
   * @param array $options
   *   (optional) An array of PDF options:
   *   - title: (string) The document title.
   *   - page_size: (string) Paper size (e.g., A4, Letter).
   *   - orientation: (string) 'portrait' or 'landscape'.
   *   - inline_css: (string) Inline CSS to include.
   *   - css_path: (string) Local or remote CSS file path/URL.
   *   - show_pagination: (bool) Whether to add page numbers.
   *   - pagination_x: (int) X coordinate for pagination text.
   *   - pagination_y: (int) Y coordinate for pagination text.
   *   - header_html: (string) HTML to insert in header.
   *   - footer_html: (string) HTML to insert in footer.
   *   - header_height_mm: (float) Header height in mm.
   *   - footer_height_mm: (float) Footer height in mm.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The generated PDF as a Symfony response object.
   */
  public function render(array $build, array $options = []): Response {
    $config = $this->configFactory->get('pdf_forge.settings');

    // Resolve base options.
    $title = (string) ($options['title'] ?? ($build['#title'] ?? 'document'));
    $pageSize = (string) ($options['page_size'] ?? $config->get('page_size') ?? 'A4');
    $orientation = (string) ($options['orientation'] ?? $config->get('orientation') ?? 'portrait');
    $inlineCss = (string) ($options['inline_css'] ?? ($build['#css'] ?? ''));
    $cssPath = $options['css_path'] ?? null;
    $showPagination = (bool) ($options['show_pagination'] ?? false);
    $paginationX = (int) ($options['pagination_x'] ?? 0);
    $paginationY = (int) ($options['pagination_y'] ?? 0);

    $headerHtml = isset($options['header_html']) ? (string) $options['header_html'] : null;
    $footerHtml = isset($options['footer_html']) ? (string) $options['footer_html'] : null;
    $headerH = isset($options['header_height_mm']) ? (float) $options['header_height_mm'] : 0.0;
    $footerH = isset($options['footer_height_mm']) ? (float) $options['footer_height_mm'] : 0.0;

    $allowRemote = (bool) $config->get('allow_remote_assets');
    $allowlist = (array) $config->get('remote_allowlist') ?: [];

    // Prepare font cache directory.
    $font_dir = self::FONT_DIR;
    $this->fileSystem->prepareDirectory(
      $font_dir,
      FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
    );
    $fontReal = $this->fileSystem->realpath($font_dir) ?: sys_get_temp_dir();

    // Configure Dompdf.
    $domOptions = new Options();
    $domOptions->set('isHtml5ParserEnabled', true);
    $domOptions->set('isRemoteEnabled', $allowRemote);
    $domOptions->set('defaultFont', (string) ($config->get('default_font') ?? 'Open Sans'));
    $domOptions->set('fontDir', $fontReal);
    $domOptions->set('fontCache', $fontReal);
    $domOptions->set('chroot', DRUPAL_ROOT);

    $dompdf = new Dompdf($domOptions);

    // Append module CSS.
    $moduleCssPath = $this->extensionPathResolver->getPath('module', 'pdf_forge') . '/css/pdf.css';
    if (is_file($moduleCssPath)) {
      $inlineCss .= "\n" . file_get_contents($moduleCssPath);
    }

    // Append custom CSS from path.
    if ($cssPath) {
      if (preg_match('@^https?://@i', (string) $cssPath)) {
        if ($allowRemote && $this->hostAllowed((string) $cssPath, $allowlist)) {
          $response = $this->httpClient->request('GET', (string) $cssPath, ['http_errors' => false]);
          $inlineCss .= "\n" . (string) $response->getBody();
        }
      }
      else {
        $resolved = (string) $cssPath;
        if (str_starts_with($resolved, 'public://') || str_starts_with($resolved, 'private://')) {
          $resolved = (string) $this->fileSystem->realpath($resolved);
        }
        if (is_file($resolved)) {
          $inlineCss .= "\n" . file_get_contents($resolved);
        }
      }
    }

    // Header/footer CSS and padding.
    $hfCss = '';
    if ($headerH > 0 || $footerH > 0) {
      $top = $headerH > 0 ? $headerH . 'mm' : '20mm';
      $bottom = $footerH > 0 ? $footerH . 'mm' : '20mm';
      $hfCss .= "@page { margin-top: {$top}; margin-bottom: {$bottom}; }\n";
      if ($headerH > 0) {
        $hfCss .= ".pdf-header{position:fixed; top:0; left:0; right:0; height: {$headerH}mm; }\n";
      }
      if ($footerH > 0) {
        $hfCss .= ".pdf-footer{position:fixed; bottom:0; left:0; right:0; height: {$footerH}mm; }\n";
      }
      $padTop = $headerH > 0 ? "padding-top: {$headerH}mm;" : '';
      $padBottom = $footerH > 0 ? "padding-bottom: {$footerH}mm;" : '';
      $hfCss .= ".pdf-body{ {$padTop} {$padBottom} }\n";
    }
    $inlineCss = $hfCss . $inlineCss;

    // Convert Drupal stream-wrapper URIs to file paths.
    if ($headerHtml) {
      $headerHtml = $this->resolveLocalUris($headerHtml);
    }
    if ($footerHtml) {
      $footerHtml = $this->resolveLocalUris($footerHtml);
    }

    // Inject variables into render array.
    $build['#css'] = $inlineCss;
    $build['#header_html'] = $headerHtml;
    $build['#footer_html'] = $footerHtml;
    if (!isset($build['#theme'])) {
      $build['#theme'] = 'pdf_forge_print';
    }
    if (!isset($build['#title'])) {
      $build['#title'] = $title;
    }

    $html = (string) $this->renderer->renderRoot($build);

    $dompdf->loadHtml($html);
    $dompdf->setPaper($pageSize, $orientation);

    // Allow alteration before rendering.
    $this->moduleHandler->alter('pdf_forge_pre_render', $dompdf);
    $dompdf->render();

    // Optional pagination.
    if ($showPagination) {
      $canvas = $dompdf->getCanvas();
      $text = '{PAGE_NUM}/{PAGE_COUNT}';
      $canvas->page_text($paginationX, $paginationY, $text, null, 10);
      $dompdf->setCanvas($canvas);
    }

    // Allow alteration after rendering.
    $this->moduleHandler->alter('pdf_forge_post_render', $dompdf);

    // Prepare PDF response.
    $response = new Response($dompdf->output());
    $response->headers->set('Content-Type', 'application/pdf');
    $safe = strtolower(trim((string) preg_replace('#\W+#', '_', $title), '_'));
    $response->headers->set('Content-Disposition', "attachment; filename={$safe}.pdf");

    return $response;
  }

  /**
   * Returns a list of supported page sizes.
   *
   * @return array
   *   Associative array of paper sizes keyed by machine name.
   */
  public function pageSizes(): array {
    $sizes = array_keys(CPDF::$PAPER_SIZES);
    return array_combine($sizes, array_map('ucfirst', $sizes));
  }

  /**
   * Returns available page orientations.
   *
   * @return array
   *   Associative array of orientations.
   */
  public function orientations(): array {
    return [
      'portrait' => $this->t('Portrait'),
      'landscape' => $this->t('Landscape'),
    ];
  }

  /**
   * Checks if a remote URL's host is allowed.
   *
   * @param string $url
   *   The remote URL.
   * @param array $allowlist
   *   Array of allowed hosts.
   *
   * @return bool
   *   TRUE if the host is allowed, FALSE otherwise.
   */
  private function hostAllowed(string $url, array $allowlist): bool {
    $host = strtolower((string) (parse_url($url, PHP_URL_HOST) ?? ''));
    foreach ($allowlist as $allowed) {
      $allowed = strtolower(trim((string) $allowed));
      if ($allowed && ($host === $allowed || str_ends_with($host, '.' . $allowed))) {
        return true;
      }
    }
    return false;
  }

  /**
   * Replaces stream-wrapper URIs inside HTML with absolute file paths.
   *
   * @param string $html
   *   The HTML string to process.
   *
   * @return string
   *   The HTML with replaced src attributes.
   */
  private function resolveLocalUris(string $html): string {
    return preg_replace_callback(
      '/src=(["\'])(public|private):\/\/([^"\']+)\1/i',
      function (array $m) {
        $quote = $m[1];
        $scheme = $m[2];
        $rest = $m[3];
        $uri = $scheme . '://' . $rest;
        $real = $this->fileSystem->realpath($uri);
        if ($real && str_starts_with($real, DRUPAL_ROOT)) {
          return 'src=' . $quote . 'file://' . $real . $quote;
        }
        return $m[0];
      },
      $html
    );
  }

}
