<?php

namespace Drupal\pdf_forge\Service;

use Symfony\Component\HttpFoundation\Response;

/**
 * Defines the contract for PDF generation services in pdf_forge.
 *
 * Implementations should:
 * - Render Drupal render arrays into HTML.
 * - Convert HTML to PDF with configurable paper size, orientation, and styling.
 * - Support optional header/footer injection and pagination.
 */
interface PdfForgeInterface {

  /**
   * Renders a Drupal render array as a PDF and returns a binary response.
   *
   * @param array $build
   *   A Drupal render array. Typically uses the 'pdf_forge_print' theme,
   *   but can be any renderable array structure.
   * @param array $options
   *   (optional) An associative array of rendering options:
   *   - title (string): Filename (without extension) for the PDF download.
   *   - page_size (string): Paper size (e.g., 'A4', 'Letter').
   *   - orientation (string): 'portrait' or 'landscape'.
   *   - inline_css (string): Additional inline CSS to inject.
   *   - css_path (string): Local file path (stream wrapper) or remote URL
   *     (must be on the allowlist for remote assets).
   *   - show_pagination (bool): Whether to show page numbers.
   *   - pagination_x (int): X-coordinate for pagination text.
   *   - pagination_y (int): Y-coordinate for pagination text.
   *   - header_html (string): HTML content to inject as a header.
   *   - header_height_mm (float): Header height in millimeters.
   *   - footer_html (string): HTML content to inject as a footer.
   *   - footer_height_mm (float): Footer height in millimeters.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   A PDF file response with `Content-Type` set to `application/pdf` and
   *   appropriate download headers.
   */
  public function render(array $build, array $options = []): Response;

  /**
   * Returns a list of supported PDF page sizes.
   *
   * @return array
   *   An associative array where keys are machine-readable size names
   *   (e.g., 'a4', 'letter') and values are human-readable labels.
   */
  public function pageSizes(): array;

  /**
   * Returns a list of supported PDF orientations.
   *
   * @return array
   *   An associative array keyed by orientation machine name ('portrait',
   *   'landscape') with translated human-readable labels.
   */
  public function orientations(): array;

}
