<?php

namespace Drupal\pdf_metadata\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\pdf_metadata\Provider\PdfMetadataProviderManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure PDF Metadata settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The PDF metadata provider manager.
   *
   * @var \Drupal\pdf_metadata\Provider\PdfMetadataProviderManager
   */
  protected PdfMetadataProviderManager $providerManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->providerManager = $container->get('pdf_metadata.provider_manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['pdf_metadata.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'pdf_metadata_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('pdf_metadata.settings');

    $form['info'] = [
      '#type' => 'item',
      '#title' => $this->t('PDF Metadata Provider'),
      '#description' => $this->t('Choose which tool to use for writing metadata to PDF files. The availability status is shown below.'),
    ];

    $form['provider'] = [
      '#type' => 'radios',
      '#title' => $this->t('Provider'),
      '#options' => $this->getProviderOptionsWithStatus(),
      '#default_value' => $config->get('provider') ?? 'ghostscript',
      '#description' => $this->getProviderDescriptions(),
    ];

    // ExifTool configuration section.
    $form['exiftool_config'] = [
      '#type' => 'details',
      '#title' => $this->t('ExifTool Configuration'),
      '#open' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="provider"]' => ['value' => 'exiftool'],
        ],
      ],
    ];

    $form['exiftool_config']['exiftool_binary_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('ExifTool Binary Path (Optional)'),
      '#description' => $this->t('Specify a custom path to the ExifTool binary. Leave empty for automatic detection. The system will search in common locations including system PATH, vendor directory, and standard installation paths.'),
      '#default_value' => $config->get('exiftool_binary_path') ?? '',
      '#placeholder' => '/usr/bin/exiftool',
    ];

    // Show detection diagnostics.
    $exiftool_provider = $this->providerManager->getProvider('exiftool');
    if ($exiftool_provider && method_exists($exiftool_provider, 'getDetectionPaths')) {
      $detection_paths = $exiftool_provider->getDetectionPaths();
      if (!empty($detection_paths)) {
        $form['exiftool_config']['detection_results'] = [
          '#type' => 'details',
          '#title' => $this->t('Auto-Detection Results'),
          '#description' => $this->t('The following paths were checked for ExifTool binary:'),
          '#open' => FALSE,
        ];

        $rows = [];
        foreach ($detection_paths as $path_info) {
          $rows[] = [
            $path_info['label'],
            $path_info['path'],
            $path_info['available'] ? '✓ Available' : '✗ Not found/executable',
          ];
        }

        $form['exiftool_config']['detection_results']['table'] = [
          '#type' => 'table',
          '#header' => [
            $this->t('Location'),
            $this->t('Path'),
            $this->t('Status'),
          ],
          '#rows' => $rows,
          '#empty' => $this->t('No ExifTool binary detected in any location.'),
        ];
      }
    }

    $form['exiftool_config']['installation_help'] = [
      '#type' => 'details',
      '#title' => $this->t('Installation Help'),
      '#open' => FALSE,
    ];

    $form['exiftool_config']['installation_help']['content'] = [
      '#markup' => $this->getInstallationHelp(),
    ];

    // GhostScript configuration section.
    $form['ghostscript_config'] = [
      '#type' => 'details',
      '#title' => $this->t('GhostScript Configuration'),
      '#open' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="provider"]' => ['value' => 'ghostscript'],
        ],
      ],
    ];

    $form['ghostscript_config']['extra_gs_options'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Extra GhostScript Options'),
      '#description' => $this->t('Enter additional GhostScript command arguments (one per line).'),
      '#default_value' => implode("\n", $config->get('extra_gs_options') ?? []),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Get provider options with availability status indicators.
   *
   * @return array
   *   Array of provider options with status.
   */
  private function getProviderOptionsWithStatus(): array {
    $options = [];
    foreach ($this->providerManager->getProviders() as $id => $provider) {
      $label = $provider->getLabel();
      $status = $provider->isAvailable() ? ' ✓' : ' ✗';
      $options[$id] = $label . $status;
    }
    return $options;
  }

  /**
   * Get detailed descriptions for each provider.
   *
   * @return string
   *   Rendered HTML descriptions.
   */
  private function getProviderDescriptions(): string {
    $descriptions = '<hr style="margin: 1em 0;">';

    // GhostScript description.
    $gs_status = $this->providerManager->getProvider('ghostscript')->isAvailable()
      ? '<strong style="color: green;">✓ Available</strong>'
      : '<strong style="color: red;">✗ Not available</strong>';
    $descriptions .= '<div style="margin-bottom: 1.5em;">';
    $descriptions .= '<strong>GhostScript</strong> ' . $gs_status . '<br/>';
    $descriptions .= '<em>Installation:</em> Server-level command-line tool<br/>';
    $descriptions .= '<em>Behavior:</em> Traditional approach, regenerates PDF. Compatible with most PDFs but <strong>may flatten interactive form fields</strong>.<br/>';
    $descriptions .= '<em>Best for:</em> Standard PDF metadata needs without forms.<br/>';
    $descriptions .= '</div>';

    // ExifTool description.
    $exif_status = $this->providerManager->getProvider('exiftool')->isAvailable()
      ? '<strong style="color: green;">✓ Available</strong>'
      : '<strong style="color: red;">✗ Not available</strong>';
    $descriptions .= '<div style="margin-bottom: 1.5em;">';
    $descriptions .= '<strong>ExifTool</strong> ' . $exif_status . '<br/>';
    $descriptions .= '<em>Installation:</em> Server-level command-line tool or Composer package<br/>';
    $descriptions .= '<em>Behavior:</em> Modern approach, modifies metadata in-place. <strong>Preserves interactive form fields</strong> and other PDF features.<br/>';
    $descriptions .= '<em>Best for:</em> PDFs with fillable forms or when file integrity is critical.<br/>';
    $descriptions .= '</div>';

    return $descriptions;
  }

  /**
   * Get installation help text.
   *
   * @return string
   *   Rendered HTML installation help.
   */
  private function getInstallationHelp(): string {
    $help = '<div style="line-height: 1.6;">';
    $help .= '<h4>' . $this->t('System-Wide Installation (Recommended for Production)') . '</h4>';
    $help .= '<p><strong>Debian/Ubuntu:</strong><br/>';
    $help .= '<code>sudo apt-get install libimage-exiftool-perl</code></p>';
    $help .= '<p><strong>RHEL/CentOS/Fedora:</strong><br/>';
    $help .= '<code>sudo yum install perl-Image-ExifTool</code></p>';
    $help .= '<p><strong>macOS (Homebrew):</strong><br/>';
    $help .= '<code>brew install exiftool</code></p>';
    $help .= '<p><strong>macOS (MacPorts):</strong><br/>';
    $help .= '<code>sudo port install p5-image-exiftool</code></p>';

    $help .= '<h4>' . $this->t('Composer Installation (Development/Testing)') . '</h4>';
    $help .= '<p><code>composer require phpexiftool/exiftool</code></p>';
    $help .= '<p style="color: #856404; background: #fff3cd; padding: 10px; border-left: 3px solid #ffc107;">';
    $help .= '<strong>⚠ Note:</strong> Composer-installed ExifTool may not work in production environments with read-only vendor directories (e.g., Acquia, Platform.sh). System-wide installation is recommended for production.';
    $help .= '</p>';

    $help .= '<h4>' . $this->t('Manual Installation') . '</h4>';
    $help .= '<p>Download from <a href="https://exiftool.org/" target="_blank">exiftool.org</a> and install according to your platform.</p>';

    $help .= '<h4>' . $this->t('Custom Path Configuration') . '</h4>';
    $help .= '<p>If ExifTool is installed in a non-standard location, specify the full path above. The binary must be executable by the web server user.</p>';

    $help .= '</div>';

    return $help;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate custom ExifTool path if provided.
    $custom_path = $form_state->getValue('exiftool_binary_path');
    if (!empty($custom_path)) {
      if (!file_exists($custom_path)) {
        $form_state->setErrorByName('exiftool_binary_path', $this->t('The specified ExifTool binary path does not exist.'));
      }
      elseif (!is_executable($custom_path)) {
        $form_state->setErrorByName('exiftool_binary_path', $this->t('The specified ExifTool binary is not executable.'));
      }
      else {
        // Try to execute it to verify it's actually ExifTool.
        exec(escapeshellarg($custom_path) . ' -ver 2>/dev/null', $output, $return_code);
        if ($return_code !== 0) {
          $form_state->setErrorByName('exiftool_binary_path', $this->t('The specified path does not appear to be a valid ExifTool binary.'));
        }
      }
    }

    // Warn if selecting a provider that is not available.
    $selected_provider = $form_state->getValue('provider');
    $provider = $this->providerManager->getProvider($selected_provider);
    if ($provider && !$provider->isAvailable()) {
      $this->messenger()->addWarning(
        $this->t('The selected provider "@provider" is not currently available. The system will fall back to GhostScript if available.',
          ['@provider' => $provider->getLabel()]
        )
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $extra_gs_options_config = $form_state->getValue('extra_gs_options');
    $extra_gs_options = array_filter(array_map('trim', explode("\n", $extra_gs_options_config)));

    $this->configFactory->getEditable('pdf_metadata.settings')
      ->set('provider', $form_state->getValue('provider'))
      ->set('exiftool_binary_path', trim($form_state->getValue('exiftool_binary_path')))
      ->set('extra_gs_options', $extra_gs_options)
      ->save();

    parent::submitForm($form, $form_state);

    // Clear provider cache after config change.
    drupal_flush_all_caches();
  }

}
