<?php

namespace Drupal\pdf_metadata\Provider;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * PDF metadata provider using ExifTool command-line tool.
 *
 * ExifTool modifies metadata in-place without regenerating the PDF,
 * which preserves interactive form fields and other PDF features.
 *
 * This provider supports multiple installation methods:
 * - System-wide ExifTool (recommended for production)
 * - Composer-installed phpexiftool/exiftool package
 * - Custom binary path configuration
 */
class ExiftoolProvider implements PdfMetadataProviderInterface {

  use StringTranslationTrait;

  /**
   * Last error message.
   *
   * @var string
   */
  protected string $lastError = '';

  /**
   * Logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Cached binary path.
   *
   * @var string|null
   */
  protected ?string $binaryPath = NULL;

  /**
   * Cached availability status.
   *
   * @var bool|null
   */
  protected ?bool $isAvailable = NULL;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   Logger factory.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory, ConfigFactoryInterface $config_factory) {
    $this->loggerFactory = $logger_factory;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public function getLabel(): string {
    return $this->t('ExifTool (Preserves interactive forms)')->render();
  }

  /**
   * {@inheritdoc}
   */
  public function getId(): string {
    return 'exiftool';
  }

  /**
   * Detect ExifTool binary with comprehensive fallback chain.
   *
   * Attempts to locate ExifTool in the following order:
   * 1. Custom configured path (admin override)
   * 2. Composer vendor directory (vendor/phpexiftool/exiftool/exiftool)
   * 3. System PATH (system-wide installation)
   * 4. Common system paths (/usr/bin, /usr/local/bin, homebrew)
   *
   * The Composer path has higher priority than system PATH to allow
   * developers to control the ExifTool version via composer.json.
   *
   * Note: Composer bin-dir is not supported because the ExifTool Perl script
   * uses relative paths to locate its library modules, which breaks when
   * executed via symlinks.
   *
   * @return string|null
   *   Path to ExifTool binary, or NULL if not found.
   */
  protected function detectBinary(): ?string {
    // Return cached path if already detected.
    if ($this->binaryPath !== NULL) {
      return $this->binaryPath;
    }

    $config = $this->configFactory->get('pdf_metadata.settings');

    // 1. Custom configured path (highest priority).
    $custom_path = $config->get('exiftool_binary_path');
    if (!empty($custom_path) && $this->validateBinary($custom_path)) {
      $this->binaryPath = $custom_path;
      return $this->binaryPath;
    }

    // 2. Composer vendor directory (project-specific version).
    $vendor_path = DRUPAL_ROOT . '/../vendor/phpexiftool/exiftool/exiftool';
    if ($this->validateBinary($vendor_path)) {
      $this->binaryPath = $vendor_path;
      return $this->binaryPath;
    }

    // 3. System PATH (recommended for production).
    exec('which exiftool 2>/dev/null', $output, $return_code);
    if ($return_code === 0 && !empty($output[0]) && $this->validateBinary($output[0])) {
      $this->binaryPath = $output[0];
      return $this->binaryPath;
    }

    // 4. Common system installation paths.
    $common_paths = [
      '/usr/bin/exiftool',
      '/usr/local/bin/exiftool',
      '/opt/homebrew/bin/exiftool',
      '/opt/local/bin/exiftool',
    ];

    foreach ($common_paths as $path) {
      if ($this->validateBinary($path)) {
        $this->binaryPath = $path;
        return $this->binaryPath;
      }
    }

    // No valid ExifTool binary found.
    $this->loggerFactory->get('pdf_metadata')
      ->warning('ExifTool binary not found in any known location.');
    return NULL;
  }

  /**
   * Validate that a binary path exists and is executable.
   *
   * @param string $path
   *   Path to validate.
   *
   * @return bool
   *   TRUE if path is valid and executable, FALSE otherwise.
   */
  protected function validateBinary(string $path): bool {
    if (!file_exists($path)) {
      return FALSE;
    }

    if (!is_executable($path)) {
      $this->loggerFactory->get('pdf_metadata')
        ->warning('ExifTool binary found but not executable: @path', ['@path' => $path]);
      return FALSE;
    }

    // Verify it's actually ExifTool by checking version.
    // Output is intentionally unused - we only check the return code.
    exec(escapeshellarg($path) . ' -ver 2>/dev/null', $output, $return_code);
    unset($output);
    if ($return_code !== 0) {
      $this->loggerFactory->get('pdf_metadata')
        ->warning('ExifTool binary validation failed: @path', ['@path' => $path]);
      return FALSE;
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function isAvailable(): bool {
    // Return cached availability status.
    if ($this->isAvailable !== NULL) {
      return $this->isAvailable;
    }

    $binary_path = $this->detectBinary();
    $this->isAvailable = ($binary_path !== NULL);

    return $this->isAvailable;
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailabilityError(): string {
    if (!$this->isAvailable()) {
      $error = $this->t('ExifTool command-line tool is not installed or not accessible.');
      $error .= ' ' . $this->t('Install options:');
      $error .= '<br/>• ' . $this->t('System-wide: apt install libimage-exiftool-perl (Debian/Ubuntu) or brew install exiftool (macOS)');
      $error .= '<br/>• ' . $this->t('Composer: composer require phpexiftool/exiftool');
      $error .= '<br/>• ' . $this->t('Manual: download from <a href="@url" target="_blank">exiftool.org</a>', ['@url' => 'https://exiftool.org/']);
      $error .= '<br/>• ' . $this->t('Custom path: configure below if ExifTool is installed in a non-standard location');
      return (string) $error;
    }
    return '';
  }

  /**
   * Get all detection paths with their status.
   *
   * This is used for diagnostic purposes in the admin UI.
   *
   * @return array
   *   Array of paths with their detection status.
   */
  public function getDetectionPaths(): array {
    $paths = [];
    $config = $this->configFactory->get('pdf_metadata.settings');

    // Custom configured path.
    $custom_path = $config->get('exiftool_binary_path');
    if (!empty($custom_path)) {
      $paths['custom'] = [
        'label' => $this->t('Custom configured path'),
        'path' => $custom_path,
        'available' => $this->validateBinary($custom_path),
      ];
    }

    // Composer vendor.
    $vendor_path = DRUPAL_ROOT . '/../vendor/phpexiftool/exiftool/exiftool';
    $paths['vendor'] = [
      'label' => $this->t('Composer vendor'),
      'path' => $vendor_path,
      'available' => $this->validateBinary($vendor_path),
    ];

    // System PATH.
    exec('which exiftool 2>/dev/null', $output, $return_code);
    $system_path = ($return_code === 0 && !empty($output[0])) ? $output[0] : '(not detected)';
    $paths['system'] = [
      'label' => $this->t('System PATH'),
      'path' => $system_path,
      'available' => ($return_code === 0 && !empty($output[0])) ? $this->validateBinary($output[0]) : FALSE,
    ];

    // Collect already detected paths to avoid duplicates.
    $detected_paths = array_column($paths, 'path');

    // Common paths (only show if not already detected).
    $common_paths = [
      '/usr/bin/exiftool' => 'Standard Linux path',
      '/usr/local/bin/exiftool' => 'Alternative Linux path',
      '/opt/homebrew/bin/exiftool' => 'Homebrew (Apple Silicon)',
      '/opt/local/bin/exiftool' => 'MacPorts',
    ];

    foreach ($common_paths as $path => $label) {
      // Skip if path is already detected via another method.
      if (in_array($path, $detected_paths, TRUE)) {
        continue;
      }

      if (file_exists($path)) {
        $paths['common_' . md5($path)] = [
          'label' => $this->t('@label', ['@label' => $label]),
          'path' => $path,
          'available' => $this->validateBinary($path),
        ];
      }
    }

    return $paths;
  }

  /**
   * {@inheritdoc}
   */
  public function writeMetadata(string $file_path, array $metadata): bool {
    // Ensure ExifTool is available.
    $binary_path = $this->detectBinary();
    if ($binary_path === NULL) {
      $this->lastError = 'ExifTool binary not found';
      $this->loggerFactory->get('pdf_metadata')
        ->error('Cannot write PDF metadata: ExifTool binary not found');
      return FALSE;
    }

    try {
      // Build ExifTool command with metadata arguments.
      $command_parts = [escapeshellarg($binary_path)];

      // Add metadata fields if provided.
      if (!empty($metadata['title'])) {
        $command_parts[] = '-Title=' . escapeshellarg($metadata['title']);
      }

      if (!empty($metadata['author'])) {
        $command_parts[] = '-Author=' . escapeshellarg($metadata['author']);
      }

      if (!empty($metadata['subject'])) {
        $command_parts[] = '-Subject=' . escapeshellarg($metadata['subject']);
      }

      if (!empty($metadata['keywords'])) {
        $command_parts[] = '-Keywords=' . escapeshellarg($metadata['keywords']);
      }

      // Overwrite original file and suppress backup file creation.
      $command_parts[] = '-overwrite_original';

      // Add file path.
      $command_parts[] = escapeshellarg($file_path);

      // Execute command.
      $command = implode(' ', $command_parts);
      exec($command . ' 2>&1', $output, $return_code);

      if ($return_code !== 0) {
        $this->lastError = 'ExifTool command failed: ' . implode("\n", $output);
        $this->loggerFactory->get('pdf_metadata')
          ->error('ExifTool command failed with code @code: @output', [
            '@code' => $return_code,
            '@output' => implode("\n", $output),
          ]);
        return FALSE;
      }

      return TRUE;
    }
    catch (\Exception $e) {
      $this->lastError = $e->getMessage();
      $this->loggerFactory->get('pdf_metadata')
        ->error('ExifTool exception: @error', ['@error' => $this->lastError]);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getLastError(): string {
    return $this->lastError;
  }

}
