<?php

namespace Drupal\pdf_metadata\Provider;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * PDF metadata provider using Ghostscript.
 */
class GhostscriptProvider implements PdfMetadataProviderInterface {

  use StringTranslationTrait;

  /**
   * Last error message.
   *
   * @var string
   */
  protected string $lastError = '';

  /**
   * Logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * Extra GhostScript options.
   *
   * @var array
   */
  protected array $extraOptions = [];

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   Logger factory.
   * @param array $extra_options
   *   Extra GhostScript command line options.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory, array $extra_options = []) {
    $this->loggerFactory = $logger_factory;
    $this->extraOptions = $extra_options;
  }

  /**
   * {@inheritdoc}
   */
  public function getLabel(): string {
    return $this->t('GhostScript (High compatibility, may flatten forms)')->render();
  }

  /**
   * {@inheritdoc}
   */
  public function getId(): string {
    return 'ghostscript';
  }

  /**
   * {@inheritdoc}
   */
  public function isAvailable(): bool {
    exec('gs -v > /dev/null 2>&1', $output, $return_code);
    return $return_code === 0;
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailabilityError(): string {
    if (!$this->isAvailable()) {
      return (string) $this->t('GhostScript is not installed or not accessible.');
    }
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function writeMetadata(string $file_path, array $metadata): bool {
    try {
      $temp_file = sys_get_temp_dir() . '/' . basename($file_path);

      // Build GhostScript options.
      $gs_options = [
        '-dBATCH',
        '-dNOPAUSE',
        '-dSAFER',
        '-sDEVICE=pdfwrite',
        '-dCompatibilityLevel=1.4',
        '-dPrinted=false',
        '-dPDFSETTINGS=/prepress',
      ];

      // Merge with extra options, removing duplicates.
      if (!empty($this->extraOptions)) {
        $extra_option_keys = array_map(
          fn($option) => explode('=', $option)[0],
          $this->extraOptions
        );
        $gs_options = array_filter(
          $gs_options,
          fn($option) => !in_array(explode('=', $option)[0], $extra_option_keys)
        );
        $gs_options = array_merge($gs_options, $this->extraOptions);
      }

      $gs_options_str = implode(' ', $gs_options);

      // Build the GhostScript command.
      $command = 'gs -o ' . escapeshellarg($temp_file) . ' ' . $gs_options_str;
      $command .= ' -f ' . escapeshellarg($file_path) . ' -c "[';
      $command .= ' /Title (' . $this->sanitizePdfMetadata($metadata['title'] ?? '') . ')';
      $command .= ' /Author (' . $this->sanitizePdfMetadata($metadata['author'] ?? '') . ')';
      $command .= ' /Subject (' . $this->sanitizePdfMetadata($metadata['subject'] ?? '') . ')';
      $command .= ' /Keywords (' . $this->sanitizePdfMetadata($metadata['keywords'] ?? '') . ')';
      $command .= ' /DOCINFO pdfmark" 2>&1';

      // Execute command.
      exec($command, $output, $return_code);

      if ($return_code === 0) {
        rename($temp_file, $file_path);
        return TRUE;
      }
      else {
        $this->lastError = implode("\n", $output);
        $this->loggerFactory->get('pdf_metadata')
          ->error('GhostScript error (code @code): @error', [
            '@code' => $return_code,
            '@error' => $this->lastError,
          ]);
        return FALSE;
      }
    }
    catch (\Exception $e) {
      $this->lastError = $e->getMessage();
      $this->loggerFactory->get('pdf_metadata')
        ->error('GhostScript exception: @error', ['@error' => $this->lastError]);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getLastError(): string {
    return $this->lastError;
  }

  /**
   * Escape special characters for PostScript string format.
   *
   * @param string $metadata
   *   The metadata string to escape.
   *
   * @return string
   *   Escaped string.
   */
  protected function sanitizePdfMetadata(string $metadata): string {
    return addcslashes($metadata, '()\\');
  }

}
