<?php

namespace Drupal\pdf_metadata\Provider;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Manages PDF metadata providers.
 */
class PdfMetadataProviderManager {

  /**
   * The registered providers.
   *
   * @var \Drupal\pdf_metadata\Provider\PdfMetadataProviderInterface[]
   */
  protected array $providers = [];

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   Logger factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory) {
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->initializeProviders();
  }

  /**
   * Initialize available providers.
   */
  protected function initializeProviders(): void {
    $config = $this->configFactory->get('pdf_metadata.settings');
    $extra_gs_options = $config->get('extra_gs_options') ?? [];

    // Register Ghostscript provider.
    $this->registerProvider(
      new GhostscriptProvider($this->loggerFactory, $extra_gs_options)
    );

    // Register ExifTool provider.
    $this->registerProvider(
      new ExiftoolProvider($this->loggerFactory, $this->configFactory)
    );
  }

  /**
   * Register a provider.
   *
   * @param \Drupal\pdf_metadata\Provider\PdfMetadataProviderInterface $provider
   *   The provider to register.
   */
  public function registerProvider(PdfMetadataProviderInterface $provider): void {
    $this->providers[$provider->getId()] = $provider;
  }

  /**
   * Get a specific provider by ID.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return \Drupal\pdf_metadata\Provider\PdfMetadataProviderInterface|null
   *   The provider, or NULL if not found.
   */
  public function getProvider(string $provider_id): ?PdfMetadataProviderInterface {
    return $this->providers[$provider_id] ?? NULL;
  }

  /**
   * Get the active provider based on configuration.
   *
   * @return \Drupal\pdf_metadata\Provider\PdfMetadataProviderInterface
   *   The active provider, defaults to Ghostscript if configured one is not
   *   available.
   */
  public function getActiveProvider(): PdfMetadataProviderInterface {
    $config = $this->configFactory->get('pdf_metadata.settings');
    $provider_id = $config->get('provider') ?? 'ghostscript';

    $provider = $this->getProvider($provider_id);

    // If the selected provider is not available, fall back to Ghostscript.
    if ($provider && !$provider->isAvailable()) {
      $provider = $this->getProvider('ghostscript');
    }

    // If even Ghostscript is not available, return ExifTool as last resort.
    if ($provider && !$provider->isAvailable()) {
      $provider = $this->getProvider('exiftool');
    }

    return $provider ?? $this->providers['ghostscript'];
  }

  /**
   * Get all registered providers.
   *
   * @return \Drupal\pdf_metadata\Provider\PdfMetadataProviderInterface[]
   *   Array of providers indexed by ID.
   */
  public function getProviders(): array {
    return $this->providers;
  }

  /**
   * Get available providers (those that are installed on the system).
   *
   * @return \Drupal\pdf_metadata\Provider\PdfMetadataProviderInterface[]
   *   Array of available providers.
   */
  public function getAvailableProviders(): array {
    return array_filter(
      $this->providers,
      fn($provider) => $provider->isAvailable()
    );
  }

  /**
   * Get provider labels for form options.
   *
   * @return array
   *   Array of provider IDs => labels.
   */
  public function getProviderOptions(): array {
    $options = [];
    foreach ($this->providers as $id => $provider) {
      $label = $provider->getLabel();
      if (!$provider->isAvailable()) {
        $label .= ' (Not available: ' . $provider->getAvailabilityError() . ')';
      }
      $options[$id] = $label;
    }
    return $options;
  }

}
