(function($, Drupal) {
        // Track processed files to avoid duplicate processing
        const processedFiles = new Set();

        // Queue and processing state
        const pdfQueue = [];
        let isProcessing = false;

        // Process the next file in the queue
        function processNextInQueue() {
            // If already processing or queue is empty, do nothing
            if (isProcessing || pdfQueue.length === 0) {
                return;
            }

            // Mark as processing and get the next item
            isProcessing = true;
            const nextItem = pdfQueue.shift();

            console.log('Processing next PDF in queue:', nextItem.fid, 'Queue remaining:', pdfQueue.length);

            // wait for 1 second before processing the next file
            setTimeout(() => {
                processPdf(nextItem.fid, nextItem.$fileContainer);
            }, 1000);
        }

        // Function to handle the actual PDF processing
        function processPdf(fid, $fileContainer) {
            fetch(`/admin/pdf-services/queue-scan?fid=${fid}`, {
                    credentials: 'same-origin'
                })
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.error) {
                        throw new Error(data.error);
                    }

                    console.log('PDF services response:', data);

                    // Handle disabled status when processing is paused by administrator
                    if (data.status === 'disabled') {
                        const $container = $fileContainer.find('.pdf-accessibility-container');
                        $container.html(`
                            <div class="pdf-accessibility-status status-disabled">
                                <span class="status-text">${data.message || 'PDF processing is currently disabled by administrator'}</span>
                            </div>
                        `);
                        // Mark processing as complete and process next file
                        isProcessing = false;
                        processNextInQueue();
                        return;
                    }

                    let accessibilityPromise = Promise.resolve();

                    // Case 1: Complete response with existing accessibility
                    if (data.status === 'completed' && data.existing) {
                        // Update UI with accessibility results
                        if (data.accessibility) {
                            const $container = $fileContainer.find('.pdf-accessibility-container');
                            $container.html(`
                            <div class="pdf-accessibility-results">
                                ${formatAccessibilityResults(data.accessibility)}
                            </div>
                        `);

                            // Explicitly re-attach the pdf-accessibility-report behavior
                            if (Drupal.behaviors.pdfAccessibilityReport && Drupal.behaviors.pdfAccessibilityReport.attach) {
                                Drupal.behaviors.pdfAccessibilityReport.attach($container[0], Drupal.settings);
                            }
                        }
                    } else {
                        // Case 2: Need to process or mixed results
                        // Handle existing accessibility results
                        if (data.accessibilityComplete && data.accessibility) {
                            // Update UI with existing accessibility results
                            const $container = $fileContainer.find('.pdf-accessibility-container');
                            $container.html(`
                            <div class="pdf-accessibility-results">
                                ${formatAccessibilityResults(data.accessibility)}
                            </div>
                        `);

                            // Explicitly re-attach the pdf-accessibility-report behavior
                            if (Drupal.behaviors.pdfAccessibilityReport && Drupal.behaviors.pdfAccessibilityReport.attach) {
                                Drupal.behaviors.pdfAccessibilityReport.attach($container[0], Drupal.settings);
                            }
                        } else if (data.accessibilityStatusId) {
                            // Set up polling for accessibility results
                            accessibilityPromise = pollAccessibilityStatus(data.accessibilityStatusId, $fileContainer);
                        }
                    }

                    // When processing is complete, process the next file
                    accessibilityPromise.finally(() => {
                        console.log('Processing complete for', fid);
                        isProcessing = false;
                        processNextInQueue();
                    });
                })
                .catch(error => {
                    console.error('Error processing PDF:', error);
                    $fileContainer.find('.pdf-accessibility-container').html(`<div class="messages messages--error">Error processing PDF: ${error.message}</div>`);

                    // Even on error, move to the next file
                    isProcessing = false;
                    processNextInQueue();
                });
        }

        Drupal.behaviors.pdfServicesWidget = {
            attach: function(context, settings) {
                // Only modify the Ajax prototype once
                if (!Drupal.Ajax.prototype.hasOwnProperty('pdfServicesHook')) {
                    // Add a flag to prevent multiple hooks
                    Drupal.Ajax.prototype.pdfServicesHook = true;

                    // Store the original success method
                    const originalSuccess = Drupal.Ajax.prototype.success;

                    // Replace with our modified version
                    Drupal.Ajax.prototype.success = function(response, status) {
                        // Process our PDF file detection before calling original
                        if (response && Array.isArray(response)) {
                            for (let i = 0; i < response.length; i++) {
                                const command = response[i];
                                if (command.command === 'insert' && command.data) {
                                    // Call original success handler
                                    const result = originalSuccess.call(this, response, status);

                                    // Process the content after it's been inserted into the DOM
                                    setTimeout(() => {
                                        // Find all PDF files in the document that we haven't processed yet
                                        $('.js-form-managed-file.form-managed-file').each(function() {
                                            const $fileContainer = $(this);
                                            const fileContainerId = $fileContainer.attr('id');

                                            // Skip if we've already processed this container or if it's not a PDF
                                            if (!fileContainerId || processedFiles.has(fileContainerId)) {
                                                return;
                                            }

                                            // Check if this is a PDF file
                                            const $file = $fileContainer.find('.file--mime-application-pdf');
                                            const $fid = $fileContainer.find('input[name$="[fids]"]');

                                            if ($file.length && $fid.length) {
                                                const fid = $fid.val();
                                                if (fid) {
                                                    console.log('Found PDF upload, processing UI for FID:', fid, 'Container ID:', fileContainerId);
                                                    // Mark this file as processed
                                                    processedFiles.add(fileContainerId);

                                                    // Handle the PDF upload
                                                    handlePdfUpload(fid, $fileContainer);
                                                }
                                            }
                                        });
                                    }, 100); // Small delay to ensure DOM is updated

                                    return result;
                                }
                            }
                        }

                        // Call original success handler if we didn't handle it above
                        return originalSuccess.call(this, response, status);
                    };
                }
            }
        };

        function handlePdfUpload(fid, $fileContainer) {
            console.log('Setting up containers for', $fileContainer.attr('id'));

            // Find the appropriate place to insert our accessibility container
            let $targetElement = $fileContainer.find('.form-managed-file__main');

            // If not found, try alternate locations
            if (!$targetElement.length) {
                $targetElement = $fileContainer;
            }

            // Create or update accessibility container
            let $accessibilityContainer = $fileContainer.find('.pdf-accessibility-container');
            if (!$accessibilityContainer.length) {
                $accessibilityContainer = $('<div class="pdf-accessibility-container"><div class="pdf-scan-loading">Checking accessibility...</div></div>');
                $targetElement.after($accessibilityContainer);
                console.log('Created accessibility container for', $fileContainer.attr('id'));
            }

            // Verify container was created
            if ($fileContainer.find('.pdf-accessibility-container').length) {
                console.log('Container verified for', $fileContainer.attr('id'));

                // Instead of immediately processing, add to queue
                pdfQueue.push({
                    fid: fid,
                    $fileContainer: $fileContainer
                });

                // Try to process next item (will start if not already processing)
                processNextInQueue();
            } else {
                console.error('Failed to create container for', $fileContainer.attr('id'));
            }
        }

        function pollAccessibilityStatus(statusId, $fileContainer) {
            console.log('Polling accessibility status for', $fileContainer.attr('id'));
            return new Promise((resolve, reject) => {
                const checkStatus = () => {
                    fetch(`/admin/pdf-services/scan/accessibility?status_id=${statusId}`)
                        .then(response => {
                            if (!response.ok) {
                                throw new Error(`HTTP error! status: ${response.status}`);
                            }
                            return response.json();
                        })
                        .then(data => {
                            if (data.error) {
                                throw new Error(data.error);
                            }

                            // Find the container (it might have moved in the DOM)
                            let $container = $fileContainer.find('.pdf-accessibility-container');
                            if (!$container.length) {
                                // Try to find by closest file container
                                const fileId = $fileContainer.attr('id');
                                $container = $(`#${fileId}`).find('.pdf-accessibility-container');
                                if (!$container.length) {
                                    console.error('Accessibility container not found for', fileId);
                                    resolve(); // Resolve anyway to avoid blocking the queue
                                    return;
                                }
                            }

                            // Update status message while processing
                            if (data.status === 'processing') {
                                $container.html('<div class="pdf-scan-loading">Checking accessibility...</div>');
                                // Poll again in 2 seconds
                                setTimeout(checkStatus, 2000);
                                return;
                            }

                            console.log('Accessibility check complete for', $fileContainer.attr('id'), data);

                            // Update UI with results matching the server-rendered version
                            $container.html(`
                        <div class="pdf-accessibility-results">
                            ${formatAccessibilityResults(data.data)}
                        </div>
                        `);

                            // Explicitly re-attach the pdf-accessibility-report behavior to the new content
                            if (Drupal.behaviors.pdfAccessibilityReport && Drupal.behaviors.pdfAccessibilityReport.attach) {
                                Drupal.behaviors.pdfAccessibilityReport.attach($container[0], Drupal.settings);
                            }

                            // Resolve the promise as this polling is complete
                            resolve();
                        })
                        .catch(error => {
                            console.error('Error polling accessibility status:', error);
                            const $container = $fileContainer.find('.pdf-accessibility-container');
                            if ($container.length) {
                                $container.html(`
                            <div class="messages messages--error">
                                Error checking accessibility: ${error.message}
                                <br>
                                <button class="button js-retry-accessibility" data-status-id="${statusId}" data-container-id="${$fileContainer.attr('id')}">
                                    Retry Check
                                </button>
                            </div>
                            `);

                                // Attach click handler to retry button
                                $container.find('.js-retry-accessibility').once('retry-handler').on('click', function() {
                                    const statusId = $(this).data('status-id');
                                    const containerId = $(this).data('container-id');
                                    pollAccessibilityStatus(statusId, $(`#${containerId}`));
                                });
                            }
                            // Resolve anyway to avoid blocking the queue
                            resolve();
                        });
                };

                // Start the polling
                checkStatus();
            });
        }

        function formatBytes(bytes, decimals = 2) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const dm = decimals < 0 ? 0 : decimals;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
        }

        function formatAccessibilityResults(data) {
            // Handle empty or invalid results
            if (!data || (!data.summary && !data.details)) {
                return `
              <div class="pdf-accessibility-status status-fails">
                  <span class="status-text">Accessibility check completed with no results</span>
              </div>
              `;
            }

            // Format the results to match the server-rendered version
            return `
              <div class="pdf-accessibility-status ${data.passes ? 'status-passes' : 'status-fails'}">
                  Accessibility: ${data.passes ? 'Passes' : 'Needs Review'}
              </div>
              <details class="accessibility-report-details" ${!data.passes ? 'open' : ''}>
                  <summary>Accessibility Report</summary>
                  <div class="pdf-accessibility-report">
                      ${formatStats(data.summary)}
                      ${formatDetailedResults(data.details)}
                  </div>
              </details>
          `;
        }

        function formatStats(summary) {
            if (!summary || Object.keys(summary).length === 0) {
                return '';
            }

            // Create the summary content matching the Twig template
            let summaryHtml = `
              <div class="accessibility-summary">
                  <div class="summary-stats">
                      <div class="stat-item passed" role="button" tabindex="0" title="Click to toggle all passed items">
                          <span class="stat-count">${summary.Passed || 0}</span>
                          <span class="stat-label">Passed</span>
                      </div>`;

            // Only add Failed section if there are failures
            if ((summary.Failed || 0) > 0) {
                summaryHtml += `
                      <div class="stat-item failed">
                          <span class="stat-count">${summary.Failed || 0}</span>
                          <span class="stat-label">Failed</span>
                      </div>`;
            }

            // Only add Needs Review section if there are items needing review
            if ((summary['Needs manual check'] || 0) > 0) {
                summaryHtml += `
                      <div class="stat-item manual">
                          <span class="stat-count">${summary['Needs manual check'] || 0}</span>
                          <span class="stat-label">Needs Review</span>
                      </div>`;
            }

            summaryHtml += `
                  </div>`;

            // Add the description if available, otherwise use default text
            if (summary.Description) {
                summaryHtml += `
                  <div class="summary-description">
                      ${summary.Description}
                  </div>`;
            } else {
                summaryHtml += `
                  <div class="summary-description">
                      The checker found problems which may prevent the document from being fully accessible.
                  </div>`;
            }

            summaryHtml += `
              </div>`;

            return summaryHtml;
        }

        function formatDetailedResults(details) {
            if (!details || Object.keys(details).length === 0) {
                return '';
            }

            let sectionsHtml = '<div class="accessibility-details">';

            for (const [section, rules] of Object.entries(details)) {
                // Count how many passed items in this section
                const passedCount = rules.filter(rule => rule.Status === 'Passed').length;

                sectionsHtml += `<div class="report-section">
                  <h4>
                      ${section}
                      ${passedCount > 0 ? `<span class="section-passed-count">(${passedCount} passed)</span>
                      <span class="section-passed-toggle">Show passed</span>` : ''}
                  </h4>
                  <div class="rules-list">`;

              for (const rule of rules) {
                  // Map status to status class
                  const statusClass = rule.Status === 'Passed' ? 'status-passed' :
                      rule.Status === 'Failed' ? 'status-failed' :
                      rule.Status === 'Needs manual check' ? 'status-manual' : 'status-skipped';

                  // Add style to hide passed items by default
                  const display = rule.Status === 'Passed' ? 'style="display: none;"' : '';

                  sectionsHtml += `
                      <div class="rule-item ${statusClass}" ${display}>
                          <div class="rule-header">
                              <span class="rule-status">${rule.Status}</span>
                              <span class="rule-name">${rule.Rule}</span>
                          </div>
                          <div class="rule-description">
                              ${rule.Description || ''}
                              ${rule.EnhancedDescription ? `<div class="enhanced-description">${rule.EnhancedDescription}</div>` : ''}
                          </div>
                      </div>`;
              }

              sectionsHtml += `</div></div>`;
          }

          sectionsHtml += '</div>';
          return sectionsHtml;
        }

        // Add the PDF Accessibility Report behavior to handle toggling passed items
        Drupal.behaviors.pdfAccessibilityReport = {
            attach: function(context, settings) {
                // Main toggle for all passed items
                once('pdf-toggle', '.accessibility-summary .stat-item.passed', context).forEach(function(element) {
                    $(element).on('click', function() {
                        const $report = $(this).closest('.pdf-accessibility-report');
                        $report.toggleClass('show-passed');

                        // Toggle visibility of all passed items
                        $report.find('.rule-item.status-passed').toggle($report.hasClass('show-passed'));

                        // Update section toggles to match the main toggle state
                        if ($report.hasClass('show-passed')) {
                            $('.section-passed-toggle', $report).text('Hide passed');
                        } else {
                            $('.section-passed-toggle', $report).text('Show passed');
                        }
                    });
                });

                // Section toggles
                once('section-toggle', '.report-section', context).forEach(function(section) {
                    const $section = $(section);

                    // Handle section toggle clicks
                    $section.find('.section-passed-toggle').on('click', function(e) {
                        e.stopPropagation();
                        const $report = $section.closest('.pdf-accessibility-report');

                        // If we're showing just this section
                        if (!$report.hasClass('show-passed')) {
                            // Show just this section's passed items
                            $section.find('.rule-item.status-passed').toggle();

                            // Update this section's toggle text
                            const isShowing = $section.find('.rule-item.status-passed').is(':visible');
                            $(this).text(isShowing ? 'Hide passed' : 'Show passed');
                        } else {
                            // If global show is on, toggle it off and show only this section
                            $report.removeClass('show-passed');
                            $section.find('.rule-item.status-passed').show();
                            $(this).text('Hide passed');

                            // Hide other sections' passed items
                            $report.find('.report-section').not($section).find('.rule-item.status-passed').hide();
                            $report.find('.report-section').not($section).find('.section-passed-toggle').text('Show passed');
                        }
                    });
                });
            }
        };

})(jQuery, Drupal);