<?php

namespace Drupal\pdf_services\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for the PDF accessibility report page.
 */
class AccessibilityReportController extends ControllerBase {

  protected $database;
  protected $dateFormatter;
  protected $pagerManager;
  protected $requestStack;
  protected $entityTypeManager;
  protected $loggerFactory;

  public function __construct(
    Connection $database,
    DateFormatterInterface $date_formatter,
    PagerManagerInterface $pager_manager,
    RequestStack $request_stack,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->database = $database;
    $this->dateFormatter = $date_formatter;
    $this->pagerManager = $pager_manager;
    $this->requestStack = $request_stack;
    $this->entityTypeManager = $entity_type_manager;
    $this->loggerFactory = $logger_factory;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('date.formatter'),
      $container->get('pager.manager'),
      $container->get('request_stack'),
      $container->get('entity_type.manager'),
      $container->get('logger.factory')
    );
  }

  public function report() {
    $build = [];

    // Add filters
    $current_request = $this->requestStack->getCurrentRequest();
    $status_filter = $current_request->query->get('status', 'all');
    $current_path = \Drupal::service('path.current')->getPath();

    // Create a form with method="get" to ensure proper URL parameters
    $form = \Drupal::formBuilder()->getForm('\Drupal\pdf_services\Form\AccessibilityFilterForm', $status_filter);
    $build['filters'] = $form;

    // Define table header with sortable columns - use actual database column names
    $header = [
      'filename' => [
        'data' => $this->t('Filename'),
        'field' => 'fid',
      ],
      'created' => [
        'data' => $this->t('Checked'),
        'field' => 'created',
        'sort' => 'desc', // Default sort
      ],
      'status' => [
        'data' => $this->t('Status'),
        'field' => 'passes',
      ],
      'issues' => [
        'data' => $this->t('Issues'),
      ],
      'report' => [
        'data' => $this->t('Report'),
      ],
      'used_in' => [
        'data' => $this->t('Used In'),
      ],
    ];

    // Query accessibility results with sorting
    $query = $this->database->select('pdf_accessibility_result', 'a')
      ->extend('\Drupal\Core\Database\Query\TableSortExtender')
      ->fields('a');

    if ($status_filter !== 'all') {
      $query->condition('passes', $status_filter === 'passes' ? 1 : 0);
    }

    // Apply the sorting based on the table header
    $query->orderByHeader($header);

    $total = $query->countQuery()->execute()->fetchField();
    $page = $this->pagerManager->createPager($total, 20);
    $offset = $page->getCurrentPage() * 20;

    $results = $query
      ->range($offset, 20)
      ->execute();

    $rows = [];
    foreach ($results as $record) {
      $file = $this->entityTypeManager->getStorage('file')->load($record->fid);
      $referencing_entity = NULL;

      // Only try to get referencing entity if file exists
      if ($file) {
        $referencing_entity = $this->getReferencingEntity($record->fid);
      }

      $report = json_decode($record->report, TRUE);
      $summary = $report['summary'] ?? [];
      $issues = $this->countIssues($summary);

      // Render summary in the table cell
      $summary_render = [
        '#theme' => 'pdf_accessibility_email_summary', // Use the email summary template for compactness
        '#summary' => $summary,
      ];
      $summary_output = \Drupal::service('renderer')->renderPlain($summary_render);

      // Render full report in a details element
      $full_report_render = [
        '#type' => 'details',
        '#title' => $this->t('View Full Report'),
        '#open' => FALSE, // Start collapsed
        'content' => [
          '#theme' => 'pdf_accessibility_report',
          '#report' => $report,
        ],
      ];

      $rows[] = [
        $file ? $file->getFilename() : $this->t('File not found'),
        $this->dateFormatter->format($record->created, 'short'),
        $record->passes ? $this->t('Passes') : $this->t('Needs Review'),
        $issues,
        ['data' => $full_report_render], // Use details element for full report
        $referencing_entity ?: $this->t('N/A'),
      ];
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No accessibility results found.'),
    ];

    $build['pager'] = [
      '#type' => 'pager',
    ];

    return $build;
  }

  protected function getReferencingEntity($fid) {
    $query = $this->database->select('file_usage', 'fu');
    $query->fields('fu', ['type', 'id'])
      ->condition('fid', $fid)
      ->range(0, 1);

    $usage = $query->execute()->fetchAssoc();
    if ($usage) {
      try {
        $entity = $this->entityTypeManager
          ->getStorage($usage['type'])
          ->load($usage['id']);
        if ($entity && $entity->access('view')) {
          return Link::createFromRoute(
            $entity->label(),
            'entity.' . $usage['type'] . '.canonical',
            [$usage['type'] => $usage['id']]
          );
        }
      }
      catch (\Exception $e) {
        $this->loggerFactory->get('pdf_services')->error('Error loading referencing entity: @error', [
          '@error' => $e->getMessage(),
        ]);
      }
    }
    return NULL;
  }

  protected function countIssues(array $summary) {
    return ($summary['Failed'] ?? 0) +
           ($summary['Failed manually'] ?? 0) +
           ($summary['Needs manual check'] ?? 0);
  }
}
