<?php

namespace Drupal\pdf_services\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\Utility\TableSort;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for the PDF analysis report page.
 */
class AnalysisReportController extends ControllerBase {

  protected $database;
  protected $dateFormatter;
  protected $pagerManager;
  protected $requestStack;
  protected $entityTypeManager;
  protected $loggerFactory;

  public function __construct(
    Connection $database,
    DateFormatterInterface $date_formatter,
    PagerManagerInterface $pager_manager,
    RequestStack $request_stack,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->database = $database;
    $this->dateFormatter = $date_formatter;
    $this->pagerManager = $pager_manager;
    $this->requestStack = $request_stack;
    $this->entityTypeManager = $entity_type_manager;
    $this->loggerFactory = $logger_factory;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('date.formatter'),
      $container->get('pager.manager'),
      $container->get('request_stack'),
      $container->get('entity_type.manager'),
      $container->get('logger.factory')
    );
  }

  public function report() {
    $build = [];

    // Define the sortable headers
    $header = [
      'filename' => [
        'data' => $this->t('Filename'),
        'field' => 'filename',
      ],
      'timestamp' => [
        'data' => $this->t('Analyzed'),
        'field' => 'timestamp',
        'sort' => 'desc', // Default sort
      ],
      'file_size' => [
        'data' => $this->t('File Size'),
        'field' => 'file_size',
      ],
      'page_count' => [
        'data' => $this->t('Pages'),
        'field' => 'page_count',
      ],
      'avg_page_bytes' => [
        'data' => $this->t('Avg Page Size'),
        'field' => 'avg_page_bytes',
      ],
      'is_linearized' => [
        'data' => $this->t('Linearized'),
        'field' => 'is_linearized',
      ],
      'is_tagged' => [
        'data' => $this->t('Tagged'),
        'field' => 'is_tagged',
      ],
      'compressed_size' => [
        'data' => $this->t('Compressed Size'),
        'field' => 'compressed_size',
      ],
      'compression_ratio' => [
        'data' => $this->t('Compression Ratio'),
        'field' => 'compression_ratio',
      ],
      'is_encrypted' => [
        'data' => $this->t('Encrypted'),
        'field' => 'is_encrypted',
      ],
      'pdf_version' => [
        'data' => $this->t('PDF Version'),
      ],
      'document_title' => [
        'data' => $this->t('Document Title'),
      ],
      'creator' => [
        'data' => $this->t('Creator'),
      ],
      'used_in' => [
        'data' => $this->t('Used In'),
      ],
    ];

    // Get current request for sorting information
    $current_request = $this->requestStack->getCurrentRequest();

    // Build the query
    $query = $this->database->select('pdf_analysis_result', 'a')
      ->extend('\Drupal\Core\Database\Query\TableSortExtender')
      ->fields('a');

    // Add the tablesort
    $query->orderByHeader($header);

    // Count total results for pager
    $total = $query->countQuery()->execute()->fetchField();
    $page = $this->pagerManager->createPager($total, 20);
    $offset = $page->getCurrentPage() * 20;

    $results = $query
      ->range($offset, 20)
      ->execute();

    $rows = [];
    foreach ($results as $record) {
      $file = $this->entityTypeManager->getStorage('file')->load($record->fid);
      $referencing_entity = NULL;

      // Only try to get referencing entity if file exists
      if ($file) {
        $referencing_entity = $this->getReferencingEntity($file->id());
      }

      // Parse the JSON results field to extract additional metadata
      $metadata = [];
      if (!empty($record->results)) {
        try {
          $json_data = json_decode($record->results, TRUE);
          if (isset($json_data['document'])) {
            $metadata = $json_data['document'];
          }
        }
        catch (\Exception $e) {
          $this->loggerFactory->get('pdf_services')->error('Error parsing JSON results: @error', [
            '@error' => $e->getMessage(),
          ]);
        }
      }

      // Extract document title and creator from metadata
      $document_title = isset($metadata['info_dict']['Title']) ? $metadata['info_dict']['Title'] : $this->t('N/A');
      $creator = isset($metadata['info_dict']['Creator']) ? $metadata['info_dict']['Creator'] :
                (isset($metadata['info_dict']['Author']) ? $metadata['info_dict']['Author'] : $this->t('N/A'));
      $pdf_version = isset($metadata['pdf_version']) ? $metadata['pdf_version'] : $this->t('N/A');

      // Format compression ratio for display
      $compression_ratio = $record->compression_ratio ?
        number_format($record->compression_ratio, 1) . '%' : $this->t('N/A');

      // Calculate average page size
      $avg_page_bytes = $record->page_count > 0 ?
        $this->formatSize($record->avg_page_bytes) : $this->t('N/A');

      $rows[] = [
        $file ? $file->getFilename() : $this->t('File not found'),
        $this->dateFormatter->format($record->timestamp, 'short'),
        $this->formatSize($record->file_size),
        $record->page_count,
        $avg_page_bytes,
        $record->is_linearized ? $this->t('Yes') : $this->t('No'),
        $record->is_tagged ? $this->t('Yes') : $this->t('No'),
        $record->compressed_size > 0 ? $this->formatSize($record->compressed_size) : $this->t('N/A'),
        $compression_ratio,
        $record->is_encrypted ? $this->t('Yes') : $this->t('No'),
        $pdf_version,
        $document_title,
        $creator,
        $referencing_entity ?: $this->t('N/A'),
      ];
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No analysis results found.'),
      '#attributes' => [
        'class' => ['pdf-analysis-report'],
      ],
    ];

    $build['pager'] = [
      '#type' => 'pager',
    ];

    return $build;
  }

  protected function getReferencingEntity($fid) {
    $query = $this->database->select('file_usage', 'fu');
    $query->fields('fu', ['type', 'id'])
      ->condition('fid', $fid)
      ->range(0, 1);

    $usage = $query->execute()->fetchAssoc();
    if ($usage) {
      try {
        $entity = $this->entityTypeManager
          ->getStorage($usage['type'])
          ->load($usage['id']);
        if ($entity && $entity->access('view')) {
          return Link::createFromRoute(
            $entity->label(),
            'entity.' . $usage['type'] . '.canonical',
            [$usage['type'] => $usage['id']]
          );
        }
      }
      catch (\Exception $e) {
        $this->loggerFactory->get('pdf_services')->error('Error loading referencing entity: @error', [
          '@error' => $e->getMessage(),
        ]);
      }
    }
    return NULL;
  }

  protected function formatSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $power = $bytes > 0 ? floor(log($bytes, 1024)) : 0;
    return number_format($bytes / pow(1024, $power), 2) . ' ' . $units[$power];
  }
}
